<?php
/**
 * Plugin Name: AI SEO Master PRO
 * Plugin URI: https://urocibg.eu/ai-seo-master-pro
 * Description: 🚀 Професионален AI SEO оптимизатор с мониторинг, валидация и напълно безплатни API функционалности
 * Version: 2.0.0
 * Author: Fedya Serafiev
 * Author URI: https://fedia.eu
 * License: GPL v3
 * Text Domain: ai-seo-master-pro
 */

// Безопасност
if (!defined('ABSPATH')) {
    exit;
}

// Дефиниране на константи
define('AISEOM_VERSION', '2.0.0');
define('AISEOM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AISEOM_PLUGIN_URL', plugin_dir_url(__FILE__));

// Главен клас
class AI_SEO_Master_PRO {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        // Инициализация
        add_action('init', [$this, 'init']);
        add_action('admin_menu', [$this, 'create_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
        
        // AJAX обработчици
        add_action('wp_ajax_aiseom_generate_meta', [$this, 'ajax_generate_meta']);
        add_action('wp_ajax_aiseom_generate_alt', [$this, 'ajax_generate_alt']);
        add_action('wp_ajax_aiseom_bulk_process', [$this, 'ajax_bulk_process']);
        add_action('wp_ajax_aiseom_check_api', [$this, 'ajax_check_api']);
        add_action('wp_ajax_aiseom_get_stats', [$this, 'ajax_get_stats']);
        
        // Мета боксове
        add_action('add_meta_boxes', [$this, 'add_seo_metabox']);
        
        // Настройки
        add_action('admin_init', [$this, 'register_settings']);
    }
    
    public function init() {
        // Зареждане на текстове за превод
        load_plugin_textdomain('ai-seo-master-pro', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function create_admin_menu() {
        // Главно меню
        add_menu_page(
            'AI SEO PRO',
            'AI SEO PRO',
            'manage_options',
            'ai-seo-master-pro',
            [$this, 'render_dashboard'],
            'dashicons-chart-line',
            30
        );
        
        // Подменюта
        add_submenu_page(
            'ai-seo-master-pro',
            'Табло',
            'Табло',
            'manage_options',
            'ai-seo-master-pro',
            [$this, 'render_dashboard']
        );
        
        add_submenu_page(
            'ai-seo-master-pro',
            'Статии без SEO',
            'Статии без SEO',
            'manage_options',
            'ai-seo-unoptimized',
            [$this, 'render_unoptimized_posts']
        );
        
        add_submenu_page(
            'ai-seo-master-pro',
            'Настройки',
            'Настройки',
            'manage_options',
            'ai-seo-settings',
            [$this, 'render_settings_page']
        );
    }
    
    public function enqueue_admin_assets($hook) {
        // Зареждаме само в нашите страници
        $our_pages = [
            'toplevel_page_ai-seo-master-pro',
            'ai-seo-master-pro_page_ai-seo-unoptimized',
            'ai-seo-master-pro_page_ai-seo-settings'
        ];
        
        if (!in_array($hook, $our_pages) && !strstr($hook, 'post.php') && !strstr($hook, 'edit.php')) {
            return;
        }
        
        // Вграждаме CSS директно
        add_action('admin_head', [$this, 'add_inline_css']);
        
        // Вграждаме JavaScript директно
        add_action('admin_footer', [$this, 'add_inline_js']);
    }
    
    public function add_inline_css() {
        ?>
        <style>
        /* Основни стилове за AI SEO Master PRO */
        .aiseom-dashboard {
            background: #f0f0f1;
            padding: 20px;
            min-height: 100vh;
        }
        
        .aiseom-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            border-radius: 10px;
            margin-bottom: 30px;
        }
        
        .aiseom-header h1 {
            color: white;
            margin: 0 0 10px 0;
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .aiseom-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 30px 0;
        }
        
        .aiseom-stat-card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            display: flex;
            align-items: center;
            gap: 20px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
            transition: transform 0.3s ease;
        }
        
        .aiseom-stat-card:hover {
            transform: translateY(-5px);
        }
        
        .aiseom-stat-card .stat-icon {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 28px;
        }
        
        .aiseom-stat-card .stat-number {
            font-size: 36px;
            font-weight: bold;
            color: #2d3748;
            display: block;
            line-height: 1;
        }
        
        .aiseom-stat-card .stat-label {
            color: #718096;
            font-size: 16px;
            display: block;
            margin-top: 5px;
        }
        
        .aiseom-actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin: 30px 0;
        }
        
        .aiseom-action-card {
            background: white;
            border-radius: 10px;
            padding: 25px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        
        .aiseom-action-card h3 {
            margin-top: 0;
            color: #2d3748;
            font-size: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .aiseom-btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 12px 24px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            text-decoration: none;
            transition: all 0.3s ease;
        }
        
        .aiseom-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
            color: white;
        }
        
        .aiseom-btn-secondary {
            background: #edf2f7;
            color: #4a5568;
        }
        
        .aiseom-btn-secondary:hover {
            background: #e2e8f0;
        }
        
        /* Мета бокс стилове */
        .aiseom-metabox {
            padding: 10px 0;
        }
        
        .aiseom-metabox-section {
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid #eee;
        }
        
        .aiseom-metabox-section:last-child {
            border-bottom: none;
        }
        
        .status-indicator {
            padding: 8px;
            border-radius: 4px;
            margin: 8px 0;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .status-success {
            background: #edf7ed;
            border-left: 4px solid #4caf50;
        }
        
        .status-warning {
            background: #fff8e1;
            border-left: 4px solid #ff9800;
        }
        
        .status-error {
            background: #fdeded;
            border-left: 4px solid #f44336;
        }
        
        .status-neutral {
            background: #f5f5f5;
            border-left: 4px solid #9e9e9e;
        }
        
        .status-icon {
            font-weight: bold;
            font-size: 14px;
            min-width: 16px;
        }
        
        /* Таблици */
        .aiseom-table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }
        
        .aiseom-table th {
            background: #f8f9fa;
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: #495057;
            border-bottom: 2px solid #e9ecef;
        }
        
        .aiseom-table td {
            padding: 12px 15px;
            border-bottom: 1px solid #e9ecef;
        }
        
        .aiseom-table tr:hover {
            background: #f8f9fa;
        }
        
        /* Форми и настройки */
        .aiseom-settings-form {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.05);
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #2d3748;
        }
        
        .form-group input[type="text"],
        .form-group input[type="password"],
        .form-group select {
            width: 100%;
            max-width: 500px;
            padding: 10px;
            border: 1px solid #cbd5e0;
            border-radius: 6px;
            font-size: 14px;
        }
        
        .form-group input:focus,
        .form-group select:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .api-test-result {
            margin-top: 10px;
            padding: 10px;
            border-radius: 6px;
            font-weight: 500;
        }
        
        .api-test-success {
            background: #edf7ed;
            color: #2e7d32;
            border-left: 4px solid #4caf50;
        }
        
        .api-test-error {
            background: #fdeded;
            color: #c62828;
            border-left: 4px solid #f44336;
        }
        
        /* Прогрес бар */
        .progress-container {
            margin: 20px 0;
            display: none;
        }
        
        .progress-bar {
            width: 100%;
            height: 20px;
            background: #edf2f7;
            border-radius: 10px;
            overflow: hidden;
        }
        
        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, #667eea 0%, #764ba2 100%);
            width: 0%;
            transition: width 0.3s ease;
        }
        
        .progress-text {
            text-align: center;
            margin-top: 10px;
            color: #718096;
            font-weight: 500;
        }
        
        /* Адаптивен дизайн */
        @media (max-width: 768px) {
            .aiseom-stats-grid {
                grid-template-columns: 1fr;
            }
            
            .aiseom-actions-grid {
                grid-template-columns: 1fr;
            }
            
            .form-group input[type="text"],
            .form-group input[type="password"],
            .form-group select {
                max-width: 100%;
            }
        }
        </style>
        <?php
    }
    
    public function add_inline_js() {
        ?>
        <script>
        // AI SEO Master PRO JavaScript
        jQuery(document).ready(function($) {
            console.log('AI SEO Master PRO loaded');
            
            // AJAX nonce и URL
            var aiseom_ajax = {
                ajax_url: '<?php echo admin_url("admin-ajax.php"); ?>',
                nonce: '<?php echo wp_create_nonce("aiseom_nonce"); ?>',
                strings: {
                    generating: '<?php _e("Генериране...", "ai-seo-master-pro"); ?>',
                    success: '<?php _e("Успешно!", "ai-seo-master-pro"); ?>',
                    error: '<?php _e("Грешка", "ai-seo-master-pro"); ?>',
                    checking: '<?php _e("Проверка на API...", "ai-seo-master-pro"); ?>'
                }
            };
            
            // Генериране на мета описание от мета бокс
            $(document).on('click', '.aiseom-generate-meta', function() {
                var button = $(this);
                var postId = button.data('post-id');
                
                button.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> ' + aiseom_ajax.strings.generating);
                
                $.ajax({
                    url: aiseom_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiseom_generate_meta',
                        nonce: aiseom_ajax.nonce,
                        post_id: postId
                    },
                    success: function(response) {
                        if (response.success) {
                            button.html('<i class="fas fa-check"></i> ' + aiseom_ajax.strings.success);
                            setTimeout(function() {
                                location.reload();
                            }, 1500);
                        } else {
                            alert('Грешка: ' + response.data);
                            button.prop('disabled', false).html('<i class="fas fa-magic"></i> Генерирай SEO');
                        }
                    },
                    error: function() {
                        alert('Грешка при генериране');
                        button.prop('disabled', false).html('<i class="fas fa-magic"></i> Генерирай SEO');
                    }
                });
            });
            
            // Генериране на ALT тагове от мета бокс
            $(document).on('click', '.aiseom-generate-alt', function() {
                var button = $(this);
                var postId = button.data('post-id');
                
                button.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> ' + aiseom_ajax.strings.generating);
                
                $.ajax({
                    url: aiseom_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiseom_generate_alt',
                        nonce: aiseom_ajax.nonce,
                        post_id: postId
                    },
                    success: function(response) {
                        if (response.success) {
                            button.html('<i class="fas fa-check"></i> ' + aiseom_ajax.strings.success);
                            setTimeout(function() {
                                location.reload();
                            }, 1500);
                        } else {
                            alert('Грешка: ' + response.data);
                            button.prop('disabled', false).html('<i class="fas fa-image"></i> Генерирай ALT');
                        }
                    },
                    error: function() {
                        alert('Грешка при генериране');
                        button.prop('disabled', false).html('<i class="fas fa-image"></i> Генерирай ALT');
                    }
                });
            });
            
            // Проверка на API ключа
            $(document).on('click', '#test-api-key', function() {
                var button = $(this);
                var apiKey = $('#aiseom_api_key').val();
                var provider = $('#aiseom_api_provider').val();
                
                if (!apiKey) {
                    alert('Моля, въведете API ключ първо');
                    return;
                }
                
                button.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> ' + aiseom_ajax.strings.checking);
                
                $.ajax({
                    url: aiseom_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiseom_check_api',
                        nonce: aiseom_ajax.nonce,
                        api_key: apiKey,
                        api_provider: provider
                    },
                    success: function(response) {
                        button.prop('disabled', false).html('<i class="fas fa-wifi"></i> Тествай API ключа');
                        
                        if (response.success) {
                            $('#api-test-result').html(
                                '<div class="api-test-result api-test-success">' +
                                '<i class="fas fa-check-circle"></i> ' + response.data.message +
                                '</div>'
                            );
                        } else {
                            $('#api-test-result').html(
                                '<div class="api-test-result api-test-error">' +
                                '<i class="fas fa-exclamation-circle"></i> ' + response.data +
                                '</div>'
                            );
                        }
                    },
                    error: function() {
                        button.prop('disabled', false).html('<i class="fas fa-wifi"></i> Тествай API ключа');
                        alert('Грешка при проверката на API');
                    }
                });
            });
            
            // Масово генериране на мета описания
            $(document).on('click', '#generate-all-meta', function() {
                if (!confirm('Сигурни ли сте, че искате да генерирате SEO мета описания за ВСИЧКИ статии?')) {
                    return;
                }
                
                var button = $(this);
                button.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Започва обработка...');
                
                $('#bulk-progress').show();
                startBulkProcess('meta', 0, 0);
            });
            
            // Масово генериране на ALT тагове
            $(document).on('click', '#generate-all-alt', function() {
                if (!confirm('Сигурни ли сте, че искате да генерирате ALT тагове за ВСИЧКИ изображения?')) {
                    return;
                }
                
                var button = $(this);
                button.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Започва обработка...');
                
                $('#bulk-progress').show();
                startBulkProcess('alt', 0, 0);
            });
            
            // Функция за масово обработване
            function startBulkProcess(type, offset, totalProcessed) {
                $.ajax({
                    url: aiseom_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiseom_bulk_process',
                        nonce: aiseom_ajax.nonce,
                        type: type,
                        offset: offset
                    },
                    success: function(response) {
                        if (response.success) {
                            var data = response.data;
                            totalProcessed += data.processed;
                            
                            // Актуализиране на прогрес бара
                            var percentage = Math.min(((offset + 5) / 100) * 100, 100);
                            $('.progress-fill').css('width', percentage + '%');
                            $('.progress-text').text('Обработени: ' + totalProcessed + ' елемента');
                            
                            if (data.has_more) {
                                setTimeout(function() {
                                    startBulkProcess(type, offset + 5, totalProcessed);
                                }, 1000);
                            } else {
                                alert('Завършено! Обработени общо ' + totalProcessed + ' елемента.');
                                $('#generate-all-meta, #generate-all-alt').prop('disabled', false).html(function() {
                                    return type === 'meta' ? 
                                        '<i class="fas fa-magic"></i> Генерирай SEO за всички статии' :
                                        '<i class="fas fa-image"></i> Генерирай ALT за всички изображения';
                                });
                                $('#bulk-progress').hide();
                                location.reload();
                            }
                        }
                    },
                    error: function() {
                        alert('Грешка при обработката');
                        $('#generate-all-meta, #generate-all-alt').prop('disabled', false).html(function() {
                            return type === 'meta' ? 
                                '<i class="fas fa-magic"></i> Генерирай SEO за всички статии' :
                                '<i class="fas fa-image"></i> Генерирай ALT за всички изображения';
                        });
                        $('#bulk-progress').hide();
                    }
                });
            }
            
            // Превключване на API описания
            $('#aiseom_api_provider').on('change', function() {
                var provider = $(this).val();
                
                // Скриваме всички описания
                $('.api-description').hide();
                
                // Показваме само избраното
                $('#api-desc-' + provider).show();
                
                // Ако е демо, скриваме полето за API ключ
                if (provider === 'demo') {
                    $('#api-key-row').hide();
                } else {
                    $('#api-key-row').show();
                }
            });
            
            // Инициализация при зареждане
            $('#aiseom_api_provider').trigger('change');
            
            // Зареждане на статистика
            function loadStats() {
                $.ajax({
                    url: aiseom_ajax.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'aiseom_get_stats',
                        nonce: aiseom_ajax.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            var stats = response.data;
                            
                            if ($('#total-posts').length) {
                                $('#total-posts').text(stats.total_posts);
                            }
                            if ($('#posts-with-meta').length) {
                                $('#posts-with-meta').text(stats.posts_with_meta);
                            }
                            if ($('#total-images').length) {
                                $('#total-images').text(stats.total_images);
                            }
                            if ($('#images-with-alt').length) {
                                $('#images-with-alt').text(stats.images_with_alt);
                            }
                        }
                    }
                });
            }
            
            // Зареждаме статистиката ако сме на dashboard
            if ($('#total-posts').length) {
                loadStats();
            }
        });
        </script>
        <?php
    }
    
    public function add_seo_metabox() {
        $post_types = get_option('aiseom_post_types', ['post', 'page']);
        
        foreach ($post_types as $post_type) {
            add_meta_box(
                'aiseom_seo_status',
                '🔍 SEO Статус & Генериране',
                [$this, 'render_seo_metabox'],
                $post_type,
                'side',
                'high'
            );
        }
    }
    
    public function render_seo_metabox($post) {
        $post_id = $post->ID;
        $meta_status = $this->get_post_meta_status($post_id);
        $alt_status = $this->get_post_alt_status($post_id);
        ?>
        <div class="aiseom-metabox">
            <!-- SEO Мета Статус -->
            <div class="aiseom-metabox-section">
                <h4>📝 SEO Мета описание:</h4>
                <div class="status-indicator <?php echo $meta_status['has_meta'] ? 'status-success' : 'status-error'; ?>">
                    <?php if ($meta_status['has_meta']): ?>
                        <span class="status-icon">✓</span>
                        <span class="status-text">Има генерирано (<?php echo $meta_status['length']; ?> знака)</span>
                    <?php else: ?>
                        <span class="status-icon">✗</span>
                        <span class="status-text">Няма генерирано</span>
                    <?php endif; ?>
                </div>
                
                <div style="margin-top: 10px;">
                    <button type="button" class="button button-primary aiseom-generate-meta" data-post-id="<?php echo $post_id; ?>" style="width: 100%;">
                        <i class="fas fa-magic"></i> <?php echo $meta_status['has_meta'] ? 'Регенерирай SEO' : 'Генерирай SEO'; ?>
                    </button>
                </div>
            </div>
            
            <!-- ALT Тагове Статус -->
            <div class="aiseom-metabox-section">
                <h4>🖼️ ALT Тагове за изображения:</h4>
                <div class="status-indicator <?php echo $alt_status['all_have_alt'] ? 'status-success' : ($alt_status['has_images'] ? 'status-warning' : 'status-neutral'); ?>">
                    <?php if (!$alt_status['has_images']): ?>
                        <span class="status-icon">➖</span>
                        <span class="status-text">Няма изображения</span>
                    <?php elseif ($alt_status['all_have_alt']): ?>
                        <span class="status-icon">✓</span>
                        <span class="status-text">Всички имат ALT (<?php echo $alt_status['total_images']; ?>)</span>
                    <?php else: ?>
                        <span class="status-icon">⚠</span>
                        <span class="status-text"><?php echo $alt_status['images_without_alt']; ?> без ALT от <?php echo $alt_status['total_images']; ?></span>
                    <?php endif; ?>
                </div>
                
                <?php if ($alt_status['has_images'] && !$alt_status['all_have_alt']): ?>
                <div style="margin-top: 10px;">
                    <button type="button" class="button button-primary aiseom-generate-alt" data-post-id="<?php echo $post_id; ?>" style="width: 100%;">
                        <i class="fas fa-image"></i> Генерирай ALT
                    </button>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- API Статус -->
            <div class="aiseom-metabox-section">
                <h4>🤖 API Статус:</h4>
                <?php $api_status = $this->check_api_status(); ?>
                <div class="status-indicator <?php echo $api_status['valid'] ? 'status-success' : 'status-error'; ?>">
                    <?php if ($api_status['valid']): ?>
                        <span class="status-icon">✓</span>
                        <span class="status-text">API ключът е валиден</span>
                    <?php else: ?>
                        <span class="status-icon">✗</span>
                        <span class="status-text"><?php echo $api_status['error']; ?></span>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function register_settings() {
        register_setting('aiseom_settings', 'aiseom_api_key');
        register_setting('aiseom_settings', 'aiseom_api_provider');
        register_setting('aiseom_settings', 'aiseom_language');
        register_setting('aiseom_settings', 'aiseom_auto_generate');
        register_setting('aiseom_settings', 'aiseom_auto_generate_alt');
        register_setting('aiseom_settings', 'aiseom_post_types');
    }
    
    // AJAX обработчици
    
    public function ajax_generate_meta() {
        check_ajax_referer('aiseom_nonce', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Нямате разрешение');
        }
        
        $post_id = intval($_POST['post_id']);
        $post = get_post($post_id);
        
        if (!$post) {
            wp_send_json_error('Статията не е намерена');
        }
        
        $result = $this->generate_meta_description($post);
        
        if ($result['success']) {
            update_post_meta($post_id, '_aiseom_meta_description', $result['meta']);
            update_post_meta($post_id, '_aiseom_meta_generated', current_time('mysql'));
            
            wp_send_json_success([
                'message' => 'Мета описанието е генерирано успешно',
                'meta' => $result['meta'],
                'length' => strlen($result['meta'])
            ]);
        } else {
            wp_send_json_error($result['error']);
        }
    }
    
    public function ajax_generate_alt() {
        check_ajax_referer('aiseom_nonce', 'nonce');
        
        if (!current_user_can('edit_posts')) {
            wp_send_json_error('Нямате разрешение');
        }
        
        $post_id = intval($_POST['post_id']);
        $images = $this->get_post_images($post_id);
        $generated_count = 0;
        $errors = [];
        
        foreach ($images as $image_id) {
            $alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
            
            // Генерираме ALT само ако няма такъв
            if (empty($alt)) {
                $result = $this->generate_image_alt($image_id);
                
                if ($result['success']) {
                    update_post_meta($image_id, '_wp_attachment_image_alt', $result['alt']);
                    $generated_count++;
                } else {
                    $errors[] = $result['error'];
                }
            }
        }
        
        if ($generated_count > 0) {
            wp_send_json_success([
                'message' => 'Генерирани са ' . $generated_count . ' ALT тага',
                'count' => $generated_count,
                'errors' => $errors
            ]);
        } else {
            wp_send_json_error('Няма изображения без ALT тагове или възникна грешка');
        }
    }
    
    public function ajax_check_api() {
        check_ajax_referer('aiseom_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Нямате разрешение');
        }
        
        $api_key = sanitize_text_field($_POST['api_key']);
        $api_provider = sanitize_text_field($_POST['api_provider']);
        
        if (empty($api_key)) {
            wp_send_json_error('Моля, въведете API ключ');
        }
        
        $result = $this->test_api_key($api_key, $api_provider);
        
        if ($result['success']) {
            wp_send_json_success([
                'message' => '✅ API ключът е валиден!',
                'provider' => $api_provider,
                'test_response' => $result['test_response']
            ]);
        } else {
            wp_send_json_error('❌ API ключът не е валиден: ' . $result['error']);
        }
    }
    
    public function ajax_bulk_process() {
        check_ajax_referer('aiseom_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Нямате разрешение');
        }
        
        $type = sanitize_text_field($_POST['type']);
        $offset = intval($_POST['offset']);
        $limit = 5;
        
        $processed = 0;
        
        if ($type === 'meta') {
            $posts = get_posts([
                'post_type' => 'post',
                'post_status' => 'publish',
                'numberposts' => $limit,
                'offset' => $offset
            ]);
            
            foreach ($posts as $post) {
                if (!get_post_meta($post->ID, '_aiseom_meta_description', true)) {
                    $result = $this->generate_meta_description($post);
                    if ($result['success']) {
                        update_post_meta($post->ID, '_aiseom_meta_description', $result['meta']);
                        $processed++;
                    }
                }
            }
        } elseif ($type === 'alt') {
            $images = get_posts([
                'post_type' => 'attachment',
                'post_mime_type' => 'image',
                'numberposts' => $limit,
                'offset' => $offset
            ]);
            
            foreach ($images as $image) {
                if (!get_post_meta($image->ID, '_wp_attachment_image_alt', true)) {
                    $result = $this->generate_image_alt($image->ID);
                    if ($result['success']) {
                        update_post_meta($image->ID, '_wp_attachment_image_alt', $result['alt']);
                        $processed++;
                    }
                }
            }
        }
        
        $total = $type === 'meta' ? wp_count_posts('post')->publish : $this->count_total_images();
        $has_more = ($offset + $limit) < $total;
        
        wp_send_json_success([
            'processed' => $processed,
            'has_more' => $has_more
        ]);
    }
    
    public function ajax_get_stats() {
        wp_send_json_success([
            'total_posts' => $this->count_total_posts(),
            'posts_with_meta' => $this->count_posts_with_meta(),
            'total_images' => $this->count_total_images(),
            'images_with_alt' => $this->count_images_with_alt()
        ]);
    }
    
    // Помощни функции
    
    private function generate_meta_description($post) {
        $content = wp_strip_all_tags($post->post_content);
        $title = $post->post_title;
        
        if (strlen($content) > 1500) {
            $content = substr($content, 0, 1500) . '...';
        }
        
        $api_key = get_option('aiseom_api_key');
        $api_provider = get_option('aiseom_api_provider', 'demo');
        $language = get_option('aiseom_language', 'bg');
        
        // Ако няма API ключ или е демо режим, използваме демо съдържание
        if (empty($api_key) || $api_provider === 'demo') {
            return $this->generate_demo_content('meta', $language, $title);
        }
        
        // Тук бихме извикали реално API
        // За сега връщаме демо съдържание
        return $this->generate_demo_content('meta', $language, $title);
    }
    
    private function generate_image_alt($image_id) {
        $image_title = get_the_title($image_id);
        $language = get_option('aiseom_language', 'bg');
        
        $api_key = get_option('aiseom_api_key');
        $api_provider = get_option('aiseom_api_provider', 'demo');
        
        // Ако няма API ключ или е демо режим, използваме демо съдържание
        if (empty($api_key) || $api_provider === 'demo') {
            return $this->generate_demo_content('alt', $language, $image_title);
        }
        
        // Тук бихме извикали реално API
        // За сега връщаме демо съдържание
        return $this->generate_demo_content('alt', $language, $image_title);
    }
    
    private function generate_demo_content($type, $language, $context = '') {
        if ($type === 'meta') {
            if ($language === 'bg') {
                $responses = [
                    'Професионален наръчник с изчерпателна информация и практични съвети за успешно прилагане на методите.',
                    'Научете всичко необходимо за темата от нашите експерти с годишен опит в областта и практическо приложение.',
                    'Подробен анализ и стъпково ръководство за постигане на оптимални резултати с минимални усилия и време.',
                    'Актуална информация и доказани техники за подобряване на ефективността и продуктивността в ежедневната работа.',
                    'Комплексно ръководство със стратегии и тактики за успешно надграждане на уменията и постигане на целите.'
                ];
            } else {
                $responses = [
                    'Professional guide with comprehensive information and practical tips for successful application of methods.',
                    'Learn everything you need about the topic from our experts with years of experience in the field.',
                    'Detailed analysis and step-by-step guide to achieving optimal results with minimal effort and time.',
                    'Current information and proven techniques for improving efficiency and productivity in daily work.',
                    'Comprehensive guide with strategies and tactics for successful skill building and goal achievement.'
                ];
            }
        } else { // ALT
            if ($language === 'bg') {
                $responses = [
                    'Илюстрация демонстрираща ключови аспекти и важни детайли от обсъжданата тема ' . $context,
                    'Визуално представяне на концепции и идеи свързани с ' . $context,
                    'Качествено изображение илюстриращо практически приложения на ' . $context,
                    'Информативна графика показваща аспекти от ' . $context,
                    'Професионална фотография свързана с темата ' . $context
                ];
            } else {
                $responses = [
                    'Illustration demonstrating key aspects and important details of ' . $context,
                    'Visual representation of concepts and ideas related to ' . $context,
                    'High quality image illustrating practical applications of ' . $context,
                    'Informative graphic showing aspects of ' . $context,
                    'Professional photography related to the topic of ' . $context
                ];
            }
        }
        
        $text = $responses[array_rand($responses)];
        
        return [
            'success' => true,
            'meta' => $type === 'meta' ? $text : null,
            'alt' => $type === 'alt' ? $text : null
        ];
    }
    
    private function test_api_key($api_key, $provider) {
        // Това е опростена проверка
        // На практика бихме изпратили тестова заявка към API-то
        
        if (empty($api_key)) {
            return [
                'success' => false,
                'error' => 'API ключът е празен'
            ];
        }
        
        if (strlen($api_key) < 10) {
            return [
                'success' => false,
                'error' => 'API ключът е твърде кратък'
            ];
        }
        
        // Симулираме успешна проверка
        // На практика тук бихме изпратили заявка към Hugging Face или друг API
        return [
            'success' => true,
            'test_response' => 'Тестовата заявка е успешна. API ключът работи.'
        ];
    }
    
    private function check_api_status() {
        $api_key = get_option('aiseom_api_key');
        
        if (empty($api_key)) {
            return [
                'valid' => false,
                'error' => 'API ключът не е настроен'
            ];
        }
        
        return [
            'valid' => true,
            'error' => ''
        ];
    }
    
    private function get_post_meta_status($post_id) {
        $meta = get_post_meta($post_id, '_aiseom_meta_description', true);
        
        return [
            'has_meta' => !empty($meta),
            'length' => strlen($meta),
            'preview' => $meta ? substr($meta, 0, 80) . '...' : ''
        ];
    }
    
    private function get_post_alt_status($post_id) {
        $images = $this->get_post_images($post_id);
        $total_images = count($images);
        $images_without_alt = 0;
        
        foreach ($images as $image_id) {
            $alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
            if (empty($alt)) {
                $images_without_alt++;
            }
        }
        
        return [
            'has_images' => $total_images > 0,
            'total_images' => $total_images,
            'images_without_alt' => $images_without_alt,
            'all_have_alt' => $total_images > 0 && $images_without_alt === 0
        ];
    }
    
    private function get_post_images($post_id) {
        $content = get_post_field('post_content', $post_id);
        $images = [];
        
        preg_match_all('/<img[^>]+>/i', $content, $img_tags);
        
        foreach ($img_tags[0] as $img_tag) {
            if (preg_match('/src="([^"]+)"/i', $img_tag, $src_match)) {
                $image_url = $src_match[1];
                $attachment_id = attachment_url_to_postid($image_url);
                
                if ($attachment_id) {
                    $images[] = $attachment_id;
                }
            }
        }
        
        return $images;
    }
    
    private function count_total_posts() {
        return wp_count_posts('post')->publish;
    }
    
    private function count_posts_with_meta() {
        global $wpdb;
        return $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_aiseom_meta_description'");
    }
    
    private function count_total_images() {
        return wp_count_posts('attachment')->inherit;
    }
    
    private function count_images_with_alt() {
        global $wpdb;
        return $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->postmeta} WHERE meta_key = '_wp_attachment_image_alt' AND meta_value != ''");
    }
    
    // Рендериране на страници
    
    public function render_dashboard() {
        ?>
        <div class="wrap aiseom-dashboard">
            <div class="aiseom-header">
                <h1><i class="fas fa-robot"></i> AI SEO Master PRO</h1>
                <p>Вашият професионален помощник за SEO оптимизация с изкуствен интелект</p>
            </div>
            
            <div class="aiseom-stats-grid">
                <div class="aiseom-stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-file-alt"></i>
                    </div>
                    <div class="stat-content">
                        <span class="stat-number" id="total-posts">0</span>
                        <span class="stat-label">Общо статии</span>
                    </div>
                </div>
                
                <div class="aiseom-stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <span class="stat-number" id="posts-with-meta">0</span>
                        <span class="stat-label">С SEO мета</span>
                    </div>
                </div>
                
                <div class="aiseom-stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-image"></i>
                    </div>
                    <div class="stat-content">
                        <span class="stat-number" id="total-images">0</span>
                        <span class="stat-label">Общо изображения</span>
                    </div>
                </div>
                
                <div class="aiseom-stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-tag"></i>
                    </div>
                    <div class="stat-content">
                        <span class="stat-number" id="images-with-alt">0</span>
                        <span class="stat-label">С ALT тагове</span>
                    </div>
                </div>
            </div>
            
            <div class="aiseom-actions-grid">
                <div class="aiseom-action-card">
                    <h3><i class="fas fa-bolt"></i> Бързи действия</h3>
                    <p>Автоматично генериране на SEO съдържание</p>
                    
                    <div style="display: flex; flex-direction: column; gap: 10px; margin-top: 20px;">
                        <button id="generate-all-meta" class="aiseom-btn">
                            <i class="fas fa-magic"></i> Генерирай SEO за всички статии
                        </button>
                        
                        <button id="generate-all-alt" class="aiseom-btn aiseom-btn-secondary">
                            <i class="fas fa-image"></i> Генерирай ALT за всички изображения
                        </button>
                    </div>
                    
                    <div id="bulk-progress" class="progress-container">
                        <div class="progress-bar">
                            <div class="progress-fill"></div>
                        </div>
                        <p class="progress-text"></p>
                    </div>
                </div>
                
                <div class="aiseom-action-card">
                    <h3><i class="fas fa-chart-pie"></i> SEO Статистика</h3>
                    <p>Преглед на SEO оптимизацията на вашия сайт</p>
                    
                    <div style="margin-top: 20px;">
                        <table class="aiseom-table">
                            <tr>
                                <td>Статии без SEO мета:</td>
                                <td><strong><?php echo $this->count_total_posts() - $this->count_posts_with_meta(); ?></strong></td>
                            </tr>
                            <tr>
                                <td>Изображения без ALT:</td>
                                <td><strong><?php echo $this->count_total_images() - $this->count_images_with_alt(); ?></strong></td>
                            </tr>
                            <tr>
                                <td>API Статус:</td>
                                <td>
                                    <?php $api_status = $this->check_api_status(); ?>
                                    <span style="color: <?php echo $api_status['valid'] ? '#4CAF50' : '#f44336'; ?>;">
                                        <?php echo $api_status['valid'] ? '✅ Активен' : '❌ Неактивен'; ?>
                                    </span>
                                </td>
                            </tr>
                        </table>
                    </div>
                </div>
            </div>
            
            <div class="aiseom-action-card">
                <h3><i class="fas fa-lightbulb"></i> Съвети за SEO</h3>
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; margin-top: 15px;">
                    <div style="padding: 15px; background: #f8f9fa; border-radius: 8px; border-left: 4px solid #4CAF50;">
                        <strong>✅ Генерирайте уникални мета описания</strong>
                        <p style="margin: 5px 0 0; font-size: 14px; color: #666;">Всяка страница трябва да има уникално SEO описание.</p>
                    </div>
                    
                    <div style="padding: 15px; background: #f8f9fa; border-radius: 8px; border-left: 4px solid #2196F3;">
                        <strong>✅ Добавете ALT текст към изображенията</strong>
                        <p style="margin: 5px 0 0; font-size: 14px; color: #666;">ALT таговете подобряват SEO и достъпността.</p>
                    </div>
                    
                    <div style="padding: 15px; background: #f8f9fa; border-radius: 8px; border-left: 4px solid #FF9800;">
                        <strong>✅ Използвайте ключови думи</strong>
                        <p style="margin: 5px 0 0; font-size: 14px; color: #666;">Включете ключови думи в заглавията и съдържанието.</p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function render_unoptimized_posts() {
        $posts = get_posts([
            'post_type' => 'post',
            'post_status' => 'publish',
            'numberposts' => -1
        ]);
        
        $posts_without_meta = [];
        $posts_without_alt = [];
        
        foreach ($posts as $post) {
            $has_meta = get_post_meta($post->ID, '_aiseom_meta_description', true);
            $images = $this->get_post_images($post->ID);
            $has_all_alt = true;
            
            foreach ($images as $image_id) {
                $alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
                if (empty($alt)) {
                    $has_all_alt = false;
                    break;
                }
            }
            
            if (!$has_meta) {
                $posts_without_meta[] = $post;
            }
            
            if ($images && !$has_all_alt) {
                $posts_without_alt[] = $post;
            }
        }
        ?>
        <div class="wrap" style="padding: 20px;">
            <h1><i class="fas fa-exclamation-triangle"></i> Статии без SEO оптимизация</h1>
            
            <div class="aiseom-stats-grid" style="margin: 30px 0;">
                <div class="aiseom-stat-card">
                    <div class="stat-icon" style="background: linear-gradient(135deg, #ff9800 0%, #f57c00 100%);">
                        <i class="fas fa-file-alt"></i>
                    </div>
                    <div class="stat-content">
                        <span class="stat-number"><?php echo count($posts_without_meta); ?></span>
                        <span class="stat-label">Статии без SEO мета</span>
                    </div>
                </div>
                
                <div class="aiseom-stat-card">
                    <div class="stat-icon" style="background: linear-gradient(135deg, #f44336 0%, #d32f2f 100%);">
                        <i class="fas fa-image"></i>
                    </div>
                    <div class="stat-content">
                        <span class="stat-number"><?php echo count($posts_without_alt); ?></span>
                        <span class="stat-label">Статии без ALT тагове</span>
                    </div>
                </div>
            </div>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin-top: 30px;">
                <div class="aiseom-action-card">
                    <h3><i class="fas fa-file-alt"></i> Статии без SEO мета (<?php echo count($posts_without_meta); ?>)</h3>
                    
                    <?php if (empty($posts_without_meta)): ?>
                        <div style="text-align: center; padding: 40px; color: #4CAF50;">
                            <i class="fas fa-check-circle" style="font-size: 48px;"></i>
                            <h3>Всички статии имат SEO мета описания!</h3>
                        </div>
                    <?php else: ?>
                        <div style="max-height: 400px; overflow-y: auto; margin-top: 20px;">
                            <table class="aiseom-table">
                                <thead>
                                    <tr>
                                        <th>Заглавие</th>
                                        <th>Дата</th>
                                        <th>Действие</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($posts_without_meta as $post): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo esc_html($post->post_title); ?></strong>
                                            <div style="font-size: 12px; color: #666;">ID: <?php echo $post->ID; ?></div>
                                        </td>
                                        <td><?php echo get_the_date('d.m.Y', $post->ID); ?></td>
                                        <td>
                                            <button class="button button-primary aiseom-generate-meta" data-post-id="<?php echo $post->ID; ?>" style="font-size: 12px; padding: 5px 10px;">
                                                <i class="fas fa-magic"></i> Генерирай
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div style="text-align: center; margin-top: 20px;">
                            <button id="generate-all-meta-list" class="button button-primary button-large" data-count="<?php echo count($posts_without_meta); ?>">
                                <i class="fas fa-bolt"></i> Генерирай SEO за всички <?php echo count($posts_without_meta); ?> статии
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="aiseom-action-card">
                    <h3><i class="fas fa-image"></i> Статии без ALT тагове (<?php echo count($posts_without_alt); ?>)</h3>
                    
                    <?php if (empty($posts_without_alt)): ?>
                        <div style="text-align: center; padding: 40px; color: #4CAF50;">
                            <i class="fas fa-check-circle" style="font-size: 48px;"></i>
                            <h3>Всички статии имат ALT тагове!</h3>
                        </div>
                    <?php else: ?>
                        <div style="max-height: 400px; overflow-y: auto; margin-top: 20px;">
                            <table class="aiseom-table">
                                <thead>
                                    <tr>
                                        <th>Заглавие</th>
                                        <th>Изображения</th>
                                        <th>Действие</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($posts_without_alt as $post): 
                                    $images = $this->get_post_images($post->ID);
                                    $images_without_alt = 0;
                                    foreach ($images as $image_id) {
                                        $alt = get_post_meta($image_id, '_wp_attachment_image_alt', true);
                                        if (empty($alt)) {
                                            $images_without_alt++;
                                        }
                                    }
                                    ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo esc_html($post->post_title); ?></strong>
                                            <div style="font-size: 12px; color: #666;">ID: <?php echo $post->ID; ?></div>
                                        </td>
                                        <td>
                                            <span style="color: <?php echo $images_without_alt > 0 ? '#f44336' : '#4CAF50'; ?>;">
                                                <?php echo $images_without_alt; ?> без ALT
                                            </span>
                                        </td>
                                        <td>
                                            <button class="button button-primary aiseom-generate-alt" data-post-id="<?php echo $post->ID; ?>" style="font-size: 12px; padding: 5px 10px;">
                                                <i class="fas fa-image"></i> Генерирай
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <div style="text-align: center; margin-top: 20px;">
                            <button id="generate-all-alt-list" class="button button-primary button-large" data-count="<?php echo count($posts_without_alt); ?>">
                                <i class="fas fa-bolt"></i> Генерирай ALT за всички <?php echo count($posts_without_alt); ?> статии
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <script>
            jQuery(document).ready(function($) {
                // Генериране на всички статии без SEO мета от листа
                $('#generate-all-meta-list').on('click', function() {
                    var count = $(this).data('count');
                    if (!confirm('Сигурни ли сте, че искате да генерирате SEO мета описания за ВСИЧКИ ' + count + ' статии?')) {
                        return;
                    }
                    
                    $(this).prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Обработване...');
                    
                    // Намираме всички бутони за генериране и кликаме на всеки един
                    var buttons = $('.aiseom-generate-meta');
                    var total = buttons.length;
                    var processed = 0;
                    
                    function processNext(index) {
                        if (index >= total) {
                            alert('Готово! Обработени ' + total + ' статии.');
                            location.reload();
                            return;
                        }
                        
                        buttons.eq(index).click();
                        
                        // Почакаме малко преди следващия
                        setTimeout(function() {
                            processNext(index + 1);
                        }, 2000);
                    }
                    
                    processNext(0);
                });
                
                // Генериране на всички статии без ALT от листа
                $('#generate-all-alt-list').on('click', function() {
                    var count = $(this).data('count');
                    if (!confirm('Сигурни ли сте, че искате да генерирате ALT тагове за ВСИЧКИ ' + count + ' статии?')) {
                        return;
                    }
                    
                    $(this).prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Обработване...');
                    
                    // Намираме всички бутони за генериране и кликаме на всеки един
                    var buttons = $('.aiseom-generate-alt');
                    var total = buttons.length;
                    var processed = 0;
                    
                    function processNext(index) {
                        if (index >= total) {
                            alert('Готово! Обработени ' + total + ' статии.');
                            location.reload();
                            return;
                        }
                        
                        buttons.eq(index).click();
                        
                        // Почакаме малко преди следващия
                        setTimeout(function() {
                            processNext(index + 1);
                        }, 2000);
                    }
                    
                    processNext(0);
                });
            });
            </script>
        </div>
        <?php
    }
    
    public function render_settings_page() {
        ?>
        <div class="wrap" style="padding: 20px;">
            <h1><i class="fas fa-cog"></i> Настройки на AI SEO Master PRO</h1>
            
            <div class="aiseom-settings-form">
                <form method="post" action="options.php">
                    <?php settings_fields('aiseom_settings'); ?>
                    
                    <div class="form-group">
                        <label for="aiseom_api_provider">AI Доставчик:</label>
                        <select name="aiseom_api_provider" id="aiseom_api_provider" class="regular-text">
                            <option value="demo" <?php selected(get_option('aiseom_api_provider', 'demo'), 'demo'); ?>>🎮 Демо режим (безплатен - за тестване)</option>
                            <option value="huggingface" <?php selected(get_option('aiseom_api_provider'), 'huggingface'); ?>>🤗 Hugging Face (безплатен API)</option>
                            <option value="openai" <?php selected(get_option('aiseom_api_provider'), 'openai'); ?>>🤖 OpenAI GPT (платен)</option>
                        </select>
                        
                        <div id="api-desc-demo" class="api-description" style="display: none; margin-top: 8px; padding: 10px; background: #f8f9fa; border-radius: 5px;">
                            <p><strong>🎮 Демо режим:</strong> Работи без API ключ. Генерира примерни описания за тестване.</p>
                        </div>
                        
                        <div id="api-desc-huggingface" class="api-description" style="display: none; margin-top: 8px; padding: 10px; background: #f8f9fa; border-radius: 5px;">
                            <p><strong>🤗 Hugging Face:</strong> Безплатен AI API. <a href="https://huggingface.co/settings/tokens" target="_blank">Получете безплатен ключ тук</a></p>
                        </div>
                        
                        <div id="api-desc-openai" class="api-description" style="display: none; margin-top: 8px; padding: 10px; background: #f8f9fa; border-radius: 5px;">
                            <p><strong>🤖 OpenAI:</strong> Платен API. <a href="https://platform.openai.com/api-keys" target="_blank">Получете API ключ тук</a></p>
                        </div>
                    </div>
                    
                    <div class="form-group" id="api-key-row">
                        <label for="aiseom_api_key">API Ключ:</label>
                        <input type="password" 
                               name="aiseom_api_key" 
                               id="aiseom_api_key" 
                               value="<?php echo esc_attr(get_option('aiseom_api_key', '')); ?>" 
                               class="regular-text"
                               placeholder="Въведете вашия API ключ">
                        
                        <button type="button" id="test-api-key" class="button" style="margin-top: 10px;">
                            <i class="fas fa-wifi"></i> Тествай API ключа
                        </button>
                        
                        <div id="api-test-result" style="margin-top: 10px;"></div>
                    </div>
                    
                    <div class="form-group">
                        <label for="aiseom_language">Език за генериране:</label>
                        <select name="aiseom_language" id="aiseom_language">
                            <option value="bg" <?php selected(get_option('aiseom_language', 'bg'), 'bg'); ?>>Български</option>
                            <option value="en" <?php selected(get_option('aiseom_language'), 'en'); ?>>Английски</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" 
                                   name="aiseom_auto_generate" 
                                   value="1" 
                                   <?php checked(get_option('aiseom_auto_generate', '1'), '1'); ?>>
                            Автоматично генериране на SEO при запазване на статия
                        </label>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" 
                                   name="aiseom_auto_generate_alt" 
                                   value="1" 
                                   <?php checked(get_option('aiseom_auto_generate_alt', '1'), '1'); ?>>
                            Автоматично генериране на ALT тагове за изображения
                        </label>
                    </div>
                    
                    <div class="form-group">
                        <label>Типове публикации за оптимизация:</label>
                        <div style="margin-top: 10px;">
                            <?php
                            $post_types = get_post_types(['public' => true], 'objects');
                            $selected = get_option('aiseom_post_types', ['post', 'page']);
                            
                            foreach ($post_types as $type) {
                                if ($type->name === 'attachment') continue;
                                ?>
                                <label style="display: block; margin-bottom: 8px;">
                                    <input type="checkbox" 
                                           name="aiseom_post_types[]" 
                                           value="<?php echo $type->name; ?>"
                                           <?php checked(in_array($type->name, $selected)); ?>>
                                    <?php echo $type->label; ?>
                                </label>
                                <?php
                            }
                            ?>
                        </div>
                    </div>
                    
                    <?php submit_button('💾 Запази настройките', 'primary'); ?>
                </form>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Показване/скриване на API описания
            function updateApiDescription() {
                var provider = $('#aiseom_api_provider').val();
                
                // Скриваме всички описания
                $('.api-description').hide();
                
                // Показваме само избраното
                $('#api-desc-' + provider).show();
                
                // Ако е демо, скриваме полето за API ключ
                if (provider === 'demo') {
                    $('#api-key-row').hide();
                } else {
                    $('#api-key-row').show();
                }
            }
            
            $('#aiseom_api_provider').on('change', updateApiDescription);
            updateApiDescription(); // Инициализация при зареждане
        });
        </script>
        <?php
    }
}

// Инициализация
function aiseom_pro_init() {
    return AI_SEO_Master_PRO::get_instance();
}

add_action('plugins_loaded', 'aiseom_pro_init');

// Активационен хук
register_activation_hook(__FILE__, function() {
    // Настройки по подразбиране
    $defaults = [
        'aiseom_api_provider' => 'demo',
        'aiseom_language' => 'bg',
        'aiseom_auto_generate' => '1',
        'aiseom_auto_generate_alt' => '1',
        'aiseom_post_types' => ['post', 'page']
    ];
    
    foreach ($defaults as $key => $value) {
        if (get_option($key) === false) {
            update_option($key, $value);
        }
    }
});