#!/bin/bash

# OpenVPN Auto-Install Script за Proxmox LXC контейнер
# Използване: sudo bash openvpn-install.sh

set -e

# Цветове за изхода
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${GREEN}╔═══════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║   OpenVPN Auto-Install за Proxmox LXC    ║${NC}"
echo -e "${GREEN}╔═══════════════════════════════════════════╗${NC}"
echo ""

# Проверка дали скриптът се изпълнява като root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}Този скрипт трябва да се изпълни като root!${NC}"
   exit 1
fi

# Проверка дали сме в LXC контейнер
if ! grep -q lxc /proc/1/cgroup 2>/dev/null && ! [ -f /.dockerenv ]; then
    echo -e "${YELLOW}Внимание: Не сте в LXC контейнер. Продължаваме...${NC}"
fi

# Конфигурационни променливи
SERVER_NAME="openvpn-server"
CLIENT_NAME="client1"
PORT=1194
PROTOCOL="udp"
DNS1="8.8.8.8"
DNS2="8.8.4.4"

# Получаване на IP адреса на сървъра
SERVER_IP=$(ip -4 addr show | grep inet | grep -v 127.0.0.1 | awk '{print $2}' | cut -d'/' -f1 | head -n1)

echo -e "${GREEN}[1/8] Конфигурация${NC}"
echo "═══════════════════════════════════════"
read -p "Въведи IP адрес на сървъра [$SERVER_IP]: " input_ip
SERVER_IP=${input_ip:-$SERVER_IP}

read -p "Въведи порт за OpenVPN [$PORT]: " input_port
PORT=${input_port:-$PORT}

read -p "Избери протокол (udp/tcp) [$PROTOCOL]: " input_protocol
PROTOCOL=${input_protocol:-$PROTOCOL}

read -p "Име на първия клиент [$CLIENT_NAME]: " input_client
CLIENT_NAME=${input_client:-$CLIENT_NAME}

echo ""
echo -e "${GREEN}[2/8] Актуализация на системата${NC}"
echo "═══════════════════════════════════════"
apt-get update -qq
apt-get upgrade -y -qq

echo ""
echo -e "${GREEN}[3/8] Инсталиране на OpenVPN и Easy-RSA${NC}"
echo "═══════════════════════════════════════"
apt-get install -y openvpn easy-rsa iptables

echo ""
echo -e "${GREEN}[4/8] Настройка на PKI и генериране на сертификати${NC}"
echo "═══════════════════════════════════════"

# Създаване на директория за Easy-RSA
make-cadir ~/openvpn-ca
cd ~/openvpn-ca

# Конфигуриране на vars файла
cat > vars <<EOF
set_var EASYRSA_REQ_COUNTRY    "BG"
set_var EASYRSA_REQ_PROVINCE   "Pazardzhik"
set_var EASYRSA_REQ_CITY       "Pazardzhik"
set_var EASYRSA_REQ_ORG        "OpenVPN-Server"
set_var EASYRSA_REQ_EMAIL      "admin@openvpn.local"
set_var EASYRSA_REQ_OU         "IT"
set_var EASYRSA_ALGO           "ec"
set_var EASYRSA_DIGEST         "sha512"
EOF

# Инициализация на PKI
./easyrsa init-pki

# Създаване на CA без парола
echo -e "\n" | ./easyrsa build-ca nopass

# Генериране на сървърен ключ и сертификат
echo -e "\n" | ./easyrsa gen-req $SERVER_NAME nopass
echo "yes" | ./easyrsa sign-req server $SERVER_NAME

# Генериране на Diffie-Hellman параметри
./easyrsa gen-dh

# Генериране на tls-auth ключ
openvpn --genkey secret pki/ta.key

# Генериране на клиентски сертификат
echo -e "\n" | ./easyrsa gen-req $CLIENT_NAME nopass
echo "yes" | ./easyrsa sign-req client $CLIENT_NAME

echo ""
echo -e "${GREEN}[5/8] Копиране на сертификатите${NC}"
echo "═══════════════════════════════════════"

# Копиране на сертификатите в OpenVPN директорията
cp pki/ca.crt /etc/openvpn/
cp pki/issued/$SERVER_NAME.crt /etc/openvpn/
cp pki/private/$SERVER_NAME.key /etc/openvpn/
cp pki/dh.pem /etc/openvpn/
cp pki/ta.key /etc/openvpn/

echo ""
echo -e "${GREEN}[6/8] Създаване на конфигурация на сървъра${NC}"
echo "═══════════════════════════════════════"

# Създаване на server.conf
cat > /etc/openvpn/server.conf <<EOF
# OpenVPN Server Configuration
port $PORT
proto $PROTOCOL
dev tun

ca ca.crt
cert $SERVER_NAME.crt
key $SERVER_NAME.key
dh dh.pem
tls-auth ta.key 0

server 10.8.0.0 255.255.255.0
ifconfig-pool-persist ipp.txt

push "redirect-gateway def1 bypass-dhcp"
push "dhcp-option DNS $DNS1"
push "dhcp-option DNS $DNS2"

keepalive 10 120
cipher AES-256-GCM
auth SHA256
user nobody
group nogroup
persist-key
persist-tun

status openvpn-status.log
log-append /var/log/openvpn.log
verb 3
explicit-exit-notify 1
EOF

echo ""
echo -e "${GREEN}[7/8] Настройка на мрежата и firewall${NC}"
echo "═══════════════════════════════════════"

# Активиране на IP forwarding
echo "net.ipv4.ip_forward=1" >> /etc/sysctl.conf
sysctl -p

# Настройка на iptables за NAT
INTERFACE=$(ip route | grep default | awk '{print $5}')
iptables -t nat -A POSTROUTING -s 10.8.0.0/24 -o $INTERFACE -j MASQUERADE

# Запазване на iptables правилата
apt-get install -y iptables-persistent
netfilter-persistent save

echo ""
echo -e "${GREEN}[8/8] Стартиране на OpenVPN${NC}"
echo "═══════════════════════════════════════"

# Активиране и стартиране на OpenVPN
systemctl enable openvpn@server
systemctl start openvpn@server

# Проверка на статуса
sleep 2
if systemctl is-active --quiet openvpn@server; then
    echo -e "${GREEN}✓ OpenVPN сървърът е стартиран успешно!${NC}"
else
    echo -e "${RED}✗ Грешка при стартиране на OpenVPN!${NC}"
    systemctl status openvpn@server
    exit 1
fi

echo ""
echo -e "${GREEN}[ФИНАЛИЗАЦИЯ] Генериране на клиентска конфигурация${NC}"
echo "═══════════════════════════════════════"

# Създаване на директория за клиентски конфигурации
mkdir -p ~/client-configs

# Създаване на базова клиентска конфигурация
cat > ~/client-configs/base.conf <<EOF
client
dev tun
proto $PROTOCOL
remote $SERVER_IP $PORT
resolv-retry infinite
nobind
user nobody
group nogroup
persist-key
persist-tun
remote-cert-tls server
cipher AES-256-GCM
auth SHA256
verb 3
EOF

# Генериране на .ovpn файл с вградени сертификати
cat > ~/client-configs/$CLIENT_NAME.ovpn <<EOF
client
dev tun
proto $PROTOCOL
remote $SERVER_IP $PORT
resolv-retry infinite
nobind
user nobody
group nogroup
persist-key
persist-tun
remote-cert-tls server
cipher AES-256-GCM
auth SHA256
verb 3
key-direction 1

<ca>
$(cat ~/openvpn-ca/pki/ca.crt)
</ca>

<cert>
$(cat ~/openvpn-ca/pki/issued/$CLIENT_NAME.crt)
</cert>

<key>
$(cat ~/openvpn-ca/pki/private/$CLIENT_NAME.key)
</key>

<tls-auth>
$(cat ~/openvpn-ca/pki/ta.key)
</tls-auth>
EOF

echo ""
echo -e "${GREEN}╔═══════════════════════════════════════════╗${NC}"
echo -e "${GREEN}║        ИНСТАЛАЦИЯТА Е ЗАВЪРШЕНА!         ║${NC}"
echo -e "${GREEN}╚═══════════════════════════════════════════╝${NC}"
echo ""
echo -e "${YELLOW}Клиентският конфигурационен файл е създаден:${NC}"
echo -e "${GREEN}~/client-configs/$CLIENT_NAME.ovpn${NC}"
echo ""
echo -e "${YELLOW}За да свалиш файла, използвай:${NC}"
echo -e "scp root@$SERVER_IP:~/client-configs/$CLIENT_NAME.ovpn ."
echo ""
echo -e "${YELLOW}Проверка на статуса:${NC}"
echo "systemctl status openvpn@server"
echo ""
echo -e "${YELLOW}Логове:${NC}"
echo "tail -f /var/log/openvpn.log"
echo ""

# Създаване на скрипт за добавяне на нови клиенти
cat > ~/add-client.sh <<'ADDCLIENT'
#!/bin/bash
# Скрипт за добавяне на нов VPN клиент

if [ -z "$1" ]; then
    echo "Използване: ./add-client.sh <име_на_клиент>"
    exit 1
fi

CLIENT_NAME=$1
cd ~/openvpn-ca

# Генериране на клиентски сертификат
echo -e "\n" | ./easyrsa gen-req $CLIENT_NAME nopass
echo "yes" | ./easyrsa sign-req client $CLIENT_NAME

# Вземане на конфигурацията
SERVER_IP=$(grep "^remote" ~/client-configs/base.conf | awk '{print $2}')
PORT=$(grep "^remote" ~/client-configs/base.conf | awk '{print $3}')
PROTOCOL=$(grep "^proto" ~/client-configs/base.conf | awk '{print $2}')

# Генериране на .ovpn файл
cat > ~/client-configs/$CLIENT_NAME.ovpn <<EOF
client
dev tun
proto $PROTOCOL
remote $SERVER_IP $PORT
resolv-retry infinite
nobind
user nobody
group nogroup
persist-key
persist-tun
remote-cert-tls server
cipher AES-256-GCM
auth SHA256
verb 3
key-direction 1

<ca>
$(cat ~/openvpn-ca/pki/ca.crt)
</ca>

<cert>
$(cat ~/openvpn-ca/pki/issued/$CLIENT_NAME.crt)
</cert>

<key>
$(cat ~/openvpn-ca/pki/private/$CLIENT_NAME.key)
</key>

<tls-auth>
$(cat ~/openvpn-ca/pki/ta.key)
</tls-auth>
EOF

echo "Клиентският файл е създаден: ~/client-configs/$CLIENT_NAME.ovpn"
ADDCLIENT

chmod +x ~/add-client.sh

echo -e "${GREEN}За добавяне на нови клиенти използвай:${NC}"
echo "~/add-client.sh <име_на_клиент>"
echo ""