#!/bin/bash

# WireGuard автоматична инсталация и настройка за Proxmox Ubuntu контейнер
# Скрипт за инсталиране на WireGuard, генериране на ключове и настройка на клиенти
# Автор: Федя Серафиев

set -e

# Цветове за по-добро визуализиране
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Функция за печатане на цветни съобщения
print_status() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_header() {
    echo -e "${BLUE}================================${NC}"
    echo -e "${BLUE}$1${NC}"
    echo -e "${BLUE}================================${NC}"
}

# Проверка дали скриптът се изпълнява като root
check_root() {
    if [[ $EUID -ne 0 ]]; then
        print_error "Този скрипт трябва да се изпълнява като root!"
        exit 1
    fi
}

# Инсталиране на WireGuard и необходими пакети
install_wireguard() {
    print_header "ИНСТАЛИРАНЕ НА WIREGUARD"
    
    # Обновяване на пакетите
    print_status "Обновяване на системата..."
    apt update && apt upgrade -y
    
    # Инсталиране на WireGuard и помощни инструменти
    print_status "Инсталиране на WireGuard и помощни инструменти..."
    apt install -y wireguard wireguard-tools qrencode iptables-persistent curl net-tools
    
    # Създаване на директории
    mkdir -p /etc/wireguard/clients
    chmod 700 /etc/wireguard
    
    print_status "WireGuard е инсталиран успешно!"
}

# Функция за получаване на външно IP
get_external_ip() {
    local external_ip=""
    local default_ip="95.42.103.123"
    
    print_status "Опит за автоматично определяне на външното IP..."
    
    # Опити за получаване на външното IP
    external_ip=$(curl -s -4 ifconfig.me 2>/dev/null || curl -s -4 icanhazip.com 2>/dev/null || curl -s -4 ipecho.net/plain 2>/dev/null || echo "")
    
    if [[ -n "$external_ip" ]]; then
        print_status "Автоматично определено външно IP: $external_ip"
        echo -e "Желаете ли да използвате това IP? ${GREEN}$external_ip${NC} (y/n) или въведете различно IP:"
        read -r user_choice
        
        if [[ "$user_choice" =~ ^[Yy]$ ]] || [[ -z "$user_choice" ]]; then
            SERVER_IP="$external_ip"
        else
            echo "Въведете външното IP адрес на сървъра:"
            read -r SERVER_IP
            if [[ -z "$SERVER_IP" ]]; then
                print_warning "Не е въведено IP. Използва се: $default_ip"
                SERVER_IP="$default_ip"
            fi
        fi
    else
        print_warning "Не може да се определи автоматично външното IP"
        echo "Въведете външното IP адрес на сървъра (или натиснете Enter за $default_ip):"
        read -r SERVER_IP
        if [[ -z "$SERVER_IP" ]]; then
            SERVER_IP="$default_ip"
        fi
    fi
    
    print_status "Ще се използва IP адрес: $SERVER_IP"
}

# Генериране на сървърски ключове
generate_server_keys() {
    print_header "ГЕНЕРИРАНЕ НА СЪРВЪРСКИ КЛЮЧОВЕ"
    
    if [[ ! -f /etc/wireguard/server_private.key ]]; then
        print_status "Генериране на сървърски частен ключ..."
        wg genkey | tee /etc/wireguard/server_private.key | wg pubkey > /etc/wireguard/server_public.key
        chmod 600 /etc/wireguard/server_private.key
        chmod 644 /etc/wireguard/server_public.key
    else
        print_status "Сървърските ключове вече съществуват."
    fi
    
    SERVER_PRIVATE_KEY=$(cat /etc/wireguard/server_private.key)
    SERVER_PUBLIC_KEY=$(cat /etc/wireguard/server_public.key)
}

# Настройка на сървърския конфигурационен файл
setup_server_config() {
    print_header "НАСТРОЙКА НА СЪРВЪРСКАТА КОНФИГУРАЦИЯ"
    
    if [[ ! -f /etc/wireguard/wg0.conf ]]; then
        print_status "Създаване на wg0.conf..."
        
        # Определяне на мрежовия интерфейс
        INTERFACE=$(ip route | grep default | head -n1 | awk '{print $5}')
        
        cat > /etc/wireguard/wg0.conf << EOF
[Interface]
PrivateKey = $SERVER_PRIVATE_KEY
Address = 10.8.0.1/24
ListenPort = 51820
SaveConfig = true
PostUp = iptables -A FORWARD -i %i -j ACCEPT; iptables -A FORWARD -o %i -j ACCEPT; iptables -t nat -A POSTROUTING -o $INTERFACE -j MASQUERADE
PostDown = iptables -D FORWARD -i %i -j ACCEPT; iptables -D FORWARD -o %i -j ACCEPT; iptables -t nat -D POSTROUTING -o $INTERFACE -j MASQUERADE

EOF
        
        print_status "Сървърската конфигурация е създадена."
    else
        print_status "Сървърската конфигурация вече съществува."
    fi
}

# Активиране на IP forwarding
enable_ip_forwarding() {
    print_status "Активиране на IP forwarding..."
    echo 'net.ipv4.ip_forward=1' >> /etc/sysctl.conf
    sysctl -p
}

# Стартиране на WireGuard службата
start_wireguard() {
    print_header "СТАРТИРАНЕ НА WIREGUARD"
    
    print_status "Стартиране и активиране на WireGuard..."
    systemctl enable wg-quick@wg0
    systemctl start wg-quick@wg0
    
    if systemctl is-active --quiet wg-quick@wg0; then
        print_status "WireGuard е стартиран успешно!"
    else
        print_error "Проблем със стартирането на WireGuard!"
        exit 1
    fi
}

# Функция за генериране на клиентски конфигурации
generate_client_config() {
    local client_name="$1"
    local client_number="$2"
    
    print_header "ГЕНЕРИРАНЕ НА КЛИЕНТ: $client_name"
    
    # Генериране на клиентски ключове
    CLIENT_PRIVATE_KEY=$(wg genkey)
    CLIENT_PUBLIC_KEY=$(echo "$CLIENT_PRIVATE_KEY" | wg pubkey)
    CLIENT_IP="10.8.0.$((client_number + 1))"
    
    # Създаване на клиентска конфигурация
    CLIENT_CONFIG="/etc/wireguard/clients/${client_name}.conf"
    
    cat > "$CLIENT_CONFIG" << EOF
[Interface]
PrivateKey = $CLIENT_PRIVATE_KEY
Address = $CLIENT_IP/32
DNS = 8.8.8.8, 1.1.1.1

[Peer]
PublicKey = $SERVER_PUBLIC_KEY
Endpoint = $SERVER_IP:51820
AllowedIPs = 0.0.0.0/0
PersistentKeepalive = 25
EOF

    print_status "Клиентската конфигурация е създадена: $CLIENT_CONFIG"
    
    # Добавяне на клиента към сървърската конфигурация
    cat >> /etc/wireguard/wg0.conf << EOF

# Client: $client_name
[Peer]
PublicKey = $CLIENT_PUBLIC_KEY
AllowedIPs = $CLIENT_IP/32
EOF

    print_status "Клиентът е добавен към сървърската конфигурация."
    
    # Рестартиране на WireGuard
    print_status "Рестартиране на WireGuard..."
    systemctl restart wg-quick@wg0
    
    # Генериране на QR код
    print_status "Генериране на QR код за $client_name..."
    qrencode -t ansiutf8 < "$CLIENT_CONFIG"
    
    # Запазване на QR код във файл
    qrencode -o "/etc/wireguard/clients/${client_name}-qr.png" < "$CLIENT_CONFIG"
    
    print_status "QR кодът е запазен като: /etc/wireguard/clients/${client_name}-qr.png"
    
    echo ""
    print_status "Конфигурационен файл за $client_name:"
    echo "================================"
    cat "$CLIENT_CONFIG"
    echo "================================"
}

# Функция за добавяне на нов клиент
add_client() {
    echo "Въведете име на клиента:"
    read -r client_name
    
    if [[ -z "$client_name" ]]; then
        print_error "Името на клиента не може да бъде празно!"
        return 1
    fi
    
    # Проверка дали клиентът вече съществува
    if [[ -f "/etc/wireguard/clients/${client_name}.conf" ]]; then
        print_warning "Клиент с име '$client_name' вече съществува!"
        return 1
    fi
    
    # Определяне на следващия номер на клиент
    client_number=$(find /etc/wireguard/clients -name "*.conf" | wc -l)
    
    # Проверка за максимален брой клиенти
    if [[ $client_number -ge 253 ]]; then
        print_error "Достигнат е максималният брой клиенти (253)!"
        return 1
    fi
    
    generate_client_config "$client_name" "$client_number"
}

# Функция за показване на статуса
show_status() {
    print_header "WIREGUARD СТАТУС"
    
    echo "Статус на услугата:"
    systemctl status wg-quick@wg0 --no-pager -l
    
    echo ""
    echo "Активни връзки:"
    wg show
    
    echo ""
    echo "Налични клиенти:"
    ls -la /etc/wireguard/clients/ 2>/dev/null || echo "Няма създадени клиенти."
}

# Главно меню
show_menu() {
    echo ""
    print_header "WIREGUARD УПРАВЛЕНИЕ"
    echo "1. Добави нов клиент"
    echo "2. Покажи статус"
    echo "3. Покажи QR код за клиент"
    echo "4. Покажи конфигурация за клиент"
    echo "5. Изход"
    echo ""
    echo -n "Изберете опция (1-5): "
}

# Функция за показване на QR код
show_qr_code() {
    echo "Налични клиенти:"
    ls /etc/wireguard/clients/*.conf 2>/dev/null | sed 's|.*/||; s|\.conf||' || { print_error "Няма създадени клиенти."; return 1; }
    
    echo ""
    echo "Въведете име на клиента за показване на QR код:"
    read -r client_name
    
    if [[ -f "/etc/wireguard/clients/${client_name}.conf" ]]; then
        qrencode -t ansiutf8 < "/etc/wireguard/clients/${client_name}.conf"
    else
        print_error "Клиент '$client_name' не съществува!"
    fi
}

# Функция за показване на конфигурация
show_client_config() {
    echo "Налични клиенти:"
    ls /etc/wireguard/clients/*.conf 2>/dev/null | sed 's|.*/||; s|\.conf||' || { print_error "Няма създадени клиенти."; return 1; }
    
    echo ""
    echo "Въведете име на клиента за показване на конфигурацията:"
    read -r client_name
    
    if [[ -f "/etc/wireguard/clients/${client_name}.conf" ]]; then
        echo "Конфигурация за клиент '$client_name':"
        echo "====================================="
        cat "/etc/wireguard/clients/${client_name}.conf"
        echo "====================================="
    else
        print_error "Клиент '$client_name' не съществува!"
    fi
}

# Главна функция
main() {
    print_header "WIREGUARD SETUP SCRIPT"
    print_status "Стартиране на автоматичната инсталация..."
    
    check_root
    
    # Проверка дали WireGuard е вече инсталиран
    if ! command -v wg &> /dev/null; then
        install_wireguard
        get_external_ip
        generate_server_keys
        setup_server_config
        enable_ip_forwarding
        start_wireguard
        
        print_status "WireGuard е инсталиран и конфигуриран успешно!"
        
        # Автоматично добавяне на първи клиент
        echo ""
        echo "Желаете ли да добавите първи клиент сега? (y/n)"
        read -r add_first_client
        if [[ "$add_first_client" =~ ^[Yy]$ ]]; then
            add_client
        fi
    else
        print_status "WireGuard е вече инсталиран."
        SERVER_IP=$(grep -E '^Endpoint' /etc/wireguard/clients/*.conf 2>/dev/null | head -1 | cut -d'=' -f2 | cut -d':' -f1 | xargs || echo "95.42.103.123")
    fi
    
    # Главен цикъл на менюто
    while true; do
        show_menu
        read -r choice
        
        case $choice in
            1)
                add_client
                ;;
            2)
                show_status
                ;;
            3)
                show_qr_code
                ;;
            4)
                show_client_config
                ;;
            5)
                print_status "Изход от скрипта."
                break
                ;;
            *)
                print_warning "Невалидна опция. Моля, изберете 1-5."
                ;;
        esac
        
        echo ""
        echo "Натиснете Enter за да продължите..."
        read -r
    done
}

# Стартиране на скрипта
main "$@"