#!/bin/bash

# Автор: Федя Серафиев

# ====== ВХОДНИ ДАННИ ОТ ПОТРЕБИТЕЛЯ ======
read -p "👉 Въведи име за контейнера на WordPress [wordpress_app]: " WP_CONTAINER
WP_CONTAINER=${WP_CONTAINER:-wordpress_app}

read -p "👉 Въведи име за контейнера на базата данни [wordpress_db]: " DB_CONTAINER
DB_CONTAINER=${DB_CONTAINER:-wordpress_db}

read -p "👉 Въведи външен порт за достъп до WordPress (напр. 8383): " HOST_PORT
HOST_PORT=${HOST_PORT:-8383}

# ====== ГЕНЕРИРАНЕ НА ПАРОЛИ ======
MYSQL_ROOT_PASSWORD=$(openssl rand -base64 16)
MYSQL_DATABASE="wordpress"
MYSQL_USER="wpuser"
MYSQL_PASSWORD=$(openssl rand -base64 16)

# ====== СЪЗДАВАНЕ НА ДИРЕКТОРИИ ======
mkdir -p config wp-content

# ====== .env ФАЙЛ ======
cat > .env <<EOF
MYSQL_ROOT_PASSWORD=${MYSQL_ROOT_PASSWORD}
MYSQL_DATABASE=${MYSQL_DATABASE}
MYSQL_USER=${MYSQL_USER}
MYSQL_PASSWORD=${MYSQL_PASSWORD}
WORDPRESS_DB_HOST=db:3306
EOF

# ====== config/php.ini ======
cat > config/php.ini <<EOF
upload_max_filesize = 64M
post_max_size = 64M
memory_limit = 256M
EOF

# ====== docker-compose.yml ======
cat > docker-compose.yml <<EOF
version: '3.8'

services:
  db:
    image: mysql:8.0
    container_name: ${DB_CONTAINER}
    restart: always
    volumes:
      - db_data:/var/lib/mysql
    environment:
      MYSQL_ROOT_PASSWORD: \${MYSQL_ROOT_PASSWORD:-rootpass}
      MYSQL_DATABASE: \${MYSQL_DATABASE:-wordpress}
      MYSQL_USER: \${MYSQL_USER:-wpuser}
      MYSQL_PASSWORD: \${MYSQL_PASSWORD:-wppass}
    healthcheck:
      test: ["CMD", "mysqladmin", "ping", "-h", "localhost"]
      interval: 10s
      timeout: 5s
      retries: 5

  wordpress:
    image: wordpress:latest
    container_name: ${WP_CONTAINER}
    depends_on:
      db:
        condition: service_healthy
    ports:
      - "${HOST_PORT}:80"
    restart: always
    volumes:
      - wordpress:/var/www/html
      - ./wp-content:/var/www/html/wp-content
      - ./config/php.ini:/usr/local/etc/php/conf.d/custom.ini
    environment:
      WORDPRESS_DB_HOST: db:3306
      WORDPRESS_DB_NAME: \${MYSQL_DATABASE:-wordpress}
      WORDPRESS_DB_USER: \${MYSQL_USER:-wpuser}
      WORDPRESS_DB_PASSWORD: \${MYSQL_PASSWORD:-wppass}

volumes:
  db_data:
    driver: local
  wordpress:
    driver: local
EOF

# ====== ИЗВОД ======
echo
echo "✅ Проектът е инициализиран успешно!"
echo "📂 Създадени: docker-compose.yml, .env, config/php.ini, wp-content/"
echo "🌐 Достъп до WordPress: http://localhost:${HOST_PORT}"
echo
echo "🔐 Данни за достъп до базата (също записани в .env):"
echo "MYSQL_ROOT_PASSWORD=${MYSQL_ROOT_PASSWORD}"
echo "MYSQL_USER=${MYSQL_USER}"
echo "MYSQL_PASSWORD=${MYSQL_PASSWORD}"
echo
echo "🚀 Стартирай с: docker compose up -d"