#!/bin/bash

# Автор: Федя Серафиев
# Сайт: https://urocibg.eu
# Дата: $(date +%Y-%m-%d)

# Установяване на Docker и Docker Compose (ако не са инсталирани)
if ! command -v docker &> /dev/null; then
    echo "Инсталиране на Docker..."
    sudo apt update
    sudo apt install -y apt-transport-https ca-certificates curl software-properties-common
    curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker-archive-keyring.gpg
    echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker-archive-keyring.gpg] https://download.docker.com/linux/ubuntu $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null
    sudo apt update
    sudo apt install -y docker-ce docker-ce-cli containerd.io
    sudo usermod -aG docker $USER
fi

if ! command -v docker-compose &> /dev/null; then
    echo "Инсталиране на Docker Compose..."
    sudo curl -L "https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
    sudo chmod +x /usr/local/bin/docker-compose
fi

# Създаване на директории за конфигурация
mkdir -p ~/cloudflared/{config,logs}

# Заявка за Cloudflare Token
read -p "Въведете вашия Cloudflare Token: " CLOUDFLARED_TOKEN
read -p "Въведете име за тунела (напр. 'my-tunnel'): " TUNNEL_NAME

# Генериране на credentials.json (ако тунелът не е създаден предварително)
if [ ! -f ~/cloudflared/config/credentials.json ]; then
    echo "Създаване на нов тунел..."
    docker run --rm cloudflare/cloudflared:latest tunnel create $TUNNEL_NAME | tee ~/cloudflared/config/credentials.json
fi

# Създаване на конфигурационен файл (config.yml)
cat > ~/cloudflared/config/config.yml <<EOL
tunnel: $TUNNEL_NAME
credentials-file: /etc/cloudflared/credentials.json
logging:
  level: info
ingress:
  - hostname: "example.yourdomain.com"  # Сменете с вашия домейн
    service: http://localhost:80       # Указател към вашия сервиз
  - service: http_status:404
EOL

# Създаване на Docker Compose файл
cat > ~/cloudflared/docker-compose.yml <<EOL
version: '3.8'
services:
  cloudflared:
    image: cloudflare/cloudflared:latest
    container_name: cloudflared-tunnel
    restart: unless-stopped
    environment:
      - TZ=Europe/Sofia
    volumes:
      - ~/cloudflared/config:/etc/cloudflared
      - ~/cloudflared/logs:/var/log/cloudflared
    command: tunnel --no-autoupdate run --token ${CLOUDFLARED_TOKEN}
EOL

# Стартиране на тунела
cd ~/cloudflared
docker-compose up -d

echo "Инсталацията завърши! Тунелът сега работи в Docker."
echo "За да добавите домейни, редактирайте ~/cloudflared/config/config.yml и рестартирайте тунела:"
echo "docker-compose restart"