#!/bin/bash

# WordPress Multi-Site Manager
# Създава изолирани WordPress инсталации с Docker

set -e

SITES_DIR="./sites"
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Цветове заOutput
GREEN='\033[0;32m'
BLUE='\033[0;34m'
RED='\033[0;31m'
NC='\033[0m'

# Генериране на силна парола
generate_password() {
    openssl rand -base64 32 | tr -d "=+/" | cut -c1-25
}

# Намиране на свободен порт
find_free_port() {
    local start_port=8080
    while netstat -tuln 2>/dev/null | grep -q ":$start_port "; do
        start_port=$((start_port + 1))
    done
    echo $start_port
}

# Създаване на нов сайт
create_site() {
    local site_name=$1

    if [ -z "$site_name" ]; then
        echo -e "${RED}Грешка: Въведи име на сайта${NC}"
        echo "Примерно: ./wp-manager.sh create mysite"
        exit 1
    fi

    local site_dir="$SITES_DIR/$site_name"

    if [ -d "$site_dir" ]; then
        echo -e "${RED}Грешка: Сайт '$site_name' вече съществува${NC}"
        exit 1
    fi

    echo -e "${BLUE}Създавам WordPress сайт: $site_name${NC}"

    # Създаване на директории
    mkdir -p "$site_dir"/{wordpress,mysql}

    # Генериране на пароли
    DB_PASSWORD=$(generate_password)
    WP_ADMIN_PASSWORD=$(generate_password)

    # Намиране на свободен порт
    HTTP_PORT=$(find_free_port)

    # Създаване на .env файл
    cat > "$site_dir/.env" << EOF
# WordPress Site: $site_name
# Създаден на: $(date)

# MySQL настройки
MYSQL_ROOT_PASSWORD=$(generate_password)
MYSQL_DATABASE=wordpress
MYSQL_USER=wpuser
MYSQL_PASSWORD=$DB_PASSWORD

# WordPress настройки
WORDPRESS_DB_HOST=db:3306
WORDPRESS_DB_NAME=wordpress
WORDPRESS_DB_USER=wpuser
WORDPRESS_DB_PASSWORD=$DB_PASSWORD

# Admin достъп
WP_ADMIN_USER=admin
WP_ADMIN_PASSWORD=$WP_ADMIN_PASSWORD
WP_ADMIN_EMAIL=admin@$site_name.local

# Портове
HTTP_PORT=$HTTP_PORT
EOF

    # Създаване на docker-compose.yml
    cat > "$site_dir/docker-compose.yml" << 'EOF'
version: '3.8'

services:
  db:
    image: mysql:8.0
    container_name: ${COMPOSE_PROJECT_NAME}_db
    restart: unless-stopped
    volumes:
      - ./mysql:/var/lib/mysql
    environment:
      MYSQL_ROOT_PASSWORD: ${MYSQL_ROOT_PASSWORD}
      MYSQL_DATABASE: ${MYSQL_DATABASE}
      MYSQL_USER: ${MYSQL_USER}
      MYSQL_PASSWORD: ${MYSQL_PASSWORD}
    networks:
      - wp_network

  wordpress:
    image: wordpress:latest
    container_name: ${COMPOSE_PROJECT_NAME}_wp
    restart: unless-stopped
    depends_on:
      - db
    ports:
      - "${HTTP_PORT}:80"
    volumes:
      - ./wordpress:/var/www/html
      - ./php.ini:/usr/local/etc/php/conf.d/custom.ini
    environment:
      WORDPRESS_DB_HOST: ${WORDPRESS_DB_HOST}
      WORDPRESS_DB_NAME: ${WORDPRESS_DB_NAME}
      WORDPRESS_DB_USER: ${WORDPRESS_DB_USER}
      WORDPRESS_DB_PASSWORD: ${WORDPRESS_DB_PASSWORD}
    networks:
      - wp_network

networks:
  wp_network:
    driver: bridge
EOF

    # Създаване на php.ini
    cat > "$site_dir/php.ini" << EOF
; Custom PHP settings
upload_max_filesize = 256M
post_max_size = 256M
memory_limit = 512M
max_execution_time = 300
max_input_time = 300
max_input_vars = 3000

; Error logging
display_errors = Off
log_errors = On
error_log = /var/log/php_errors.log

; Session
session.gc_maxlifetime = 1440
session.cookie_httponly = On

; Security
expose_php = Off
allow_url_fopen = On
EOF

    # Стартиране на контейнерите
    cd "$site_dir"
    COMPOSE_PROJECT_NAME="${site_name}" docker-compose up -d

    echo -e "${GREEN}✓ Сайт '$site_name' създаден успешно!${NC}"
    echo ""
    echo -e "${BLUE}Достъп до сайта:${NC}"
    echo "  URL: http://localhost:$HTTP_PORT"
    echo "  Admin: http://localhost:$HTTP_PORT/wp-admin"
    echo ""
    echo -e "${BLUE}Admin креденшъли:${NC}"
    echo "  Потребител: admin"
    echo "  Парола: $WP_ADMIN_PASSWORD"
    echo ""
    echo -e "${BLUE}Файлове:${NC}"
    echo "  Локация: $site_dir"
    echo "  .env файл: $site_dir/.env"
    echo ""
    echo -e "${GREEN}Изчакай 30-60 секунди за инициализация на базата${NC}"
}

# Спиране на сайт
stop_site() {
    local site_name=$1
    local site_dir="$SITES_DIR/$site_name"

    if [ ! -d "$site_dir" ]; then
        echo -e "${RED}Грешка: Сайт '$site_name' не съществува${NC}"
        exit 1
    fi

    cd "$site_dir"
    COMPOSE_PROJECT_NAME="${site_name}" docker-compose stop
    echo -e "${GREEN}✓ Сайт '$site_name' спрян${NC}"
}

# Стартиране на сайт
start_site() {
    local site_name=$1
    local site_dir="$SITES_DIR/$site_name"

    if [ ! -d "$site_dir" ]; then
        echo -e "${RED}Грешка: Сайт '$site_name' не съществува${NC}"
        exit 1
    fi

    cd "$site_dir"
    COMPOSE_PROJECT_NAME="${site_name}" docker-compose start

    source .env
    echo -e "${GREEN}✓ Сайт '$site_name' стартиран${NC}"
    echo "  URL: http://localhost:$HTTP_PORT"
}

# Изтриване на сайт
delete_site() {
    local site_name=$1
    local site_dir="$SITES_DIR/$site_name"

    if [ ! -d "$site_dir" ]; then
        echo -e "${RED}Грешка: Сайт '$site_name' не съществува${NC}"
        exit 1
    fi

    read -p "Сигурен ли си, че искаш да изтриеш '$site_name'? (yes/no): " confirm
    if [ "$confirm" != "yes" ]; then
        echo "Отказано"
        exit 0
    fi

    cd "$site_dir"
    COMPOSE_PROJECT_NAME="${site_name}" docker-compose down -v
    cd ..
    rm -rf "$site_dir"

    echo -e "${GREEN}✓ Сайт '$site_name' изтрит${NC}"
}

# Списък със сайтове
list_sites() {
    if [ ! -d "$SITES_DIR" ] || [ -z "$(ls -A $SITES_DIR 2>/dev/null)" ]; then
        echo "Няма създадени сайтове"
        exit 0
    fi

    echo -e "${BLUE}Налични WordPress сайтове:${NC}"
    echo ""

    for site_path in "$SITES_DIR"/*/ ; do
        if [ -d "$site_path" ]; then
            site_name=$(basename "$site_path")
            env_file="$site_path.env"

            if [ -f "$env_file" ]; then
                # Вземаме порта от .env
                HTTP_PORT=$(grep "^HTTP_PORT=" "$env_file" | cut -d'=' -f2)

                # Проверяваме дали контейнерът работи (без wp_ префикс)
                container_id=$(docker ps -q -f "name=${site_name}_wp" 2>/dev/null)

                if [ -n "$container_id" ]; then
                    echo -e "  ${GREEN}●${NC} $site_name - http://localhost:$HTTP_PORT"
                else
                    echo -e "  ${RED}○${NC} $site_name - http://localhost:$HTTP_PORT (спрян)"
                fi
            fi
        fi
    done
}

# Показване на информация за сайт
info_site() {
    local site_name=$1
    local site_dir="$SITES_DIR/$site_name"

    if [ ! -d "$site_dir" ]; then
        echo -e "${RED}Грешка: Сайт '$site_name' не съществува${NC}"
        exit 1
    fi

    source "$site_dir/.env"

    echo -e "${BLUE}Информация за сайт: $site_name${NC}"
    echo ""
    echo "URL: http://localhost:$HTTP_PORT"
    echo "Admin: http://localhost:$HTTP_PORT/wp-admin"
    echo ""
    echo "WordPress Admin:"
    echo "  Потребител: $WP_ADMIN_USER"
    echo "  Парола: $WP_ADMIN_PASSWORD"
    echo ""
    echo "MySQL:"
    echo "  База: $MYSQL_DATABASE"
    echo "  Потребител: $MYSQL_USER"
    echo "  Парола: $MYSQL_PASSWORD"
    echo ""
    echo "Файлове: $site_dir"
}

# Помощна информация
show_help() {
    cat << EOF
WordPress Multi-Site Manager

Използване:
  ./wp-manager.sh <команда> [опции]

Команди:
  create <име>    Създаване на нов WordPress сайт
  start <име>     Стартиране на съществуващ сайт
  stop <име>      Спиране на работещ сайт
  delete <име>    Изтриване на сайт (с потвърждение)
  list            Показване на всички сайтове
  info <име>      Информация и креденшъли за сайт
  help            Показване на тази помощ

Примери:
  ./wp-manager.sh create mysite
  ./wp-manager.sh list
  ./wp-manager.sh info mysite
  ./wp-manager.sh stop mysite
  ./wp-manager.sh delete mysite

EOF
}

# Главна логика
main() {
    mkdir -p "$SITES_DIR"

    case "${1:-}" in
        create)
            create_site "$2"
            ;;
        start)
            start_site "$2"
            ;;
        stop)
            stop_site "$2"
            ;;
        delete)
            delete_site "$2"
            ;;
        list)
            list_sites
            ;;
        info)
            info_site "$2"
            ;;
        help|--help|-h)
            show_help
            ;;
        *)
            show_help
            exit 1
            ;;
    esac
}

main "$@"