#!/bin/bash

# Настройки
LOCAL_DIR="/mnt/d/backup_proxmox"                  # Локална папка за архивите (на Windows)
REMOTE_DIR="/var/lib/vz/dump"                      # Отдалечена папка с архиви на Proxmox
REMOTE_HOST="prox"                                 # SSH алиас за Proxmox хоста (например в ~/.ssh/config)

# ---

# 1. Създаване на целевата папка, ако не съществува
echo "📂 Проверявам локалната папка: $LOCAL_DIR"
mkdir -p "$LOCAL_DIR"

if [ ! -d "$LOCAL_DIR" ]; then
    echo "❌ Грешка: Не мога да създам или достъпя локалната папка '$LOCAL_DIR'. Проверете пътя и разрешенията."
    exit 1
fi

echo "🔍 Започвам синхронизация на Proxmox LXC архиви..."

# 2. Проверка на SSH връзката преди старт на синхронизацията
echo "📡 Проверявам SSH връзка с '$REMOTE_HOST'..."
ssh -q "$REMOTE_HOST" exit
if [ $? -ne 0 ]; then
    echo "❌ Грешка: Не мога да установя SSH връзка с '$REMOTE_HOST'. Моля, проверете SSH алиаса или мрежовата свързаност."
    exit 1
fi

echo "🔄 Синхронизирам архиви от ${REMOTE_HOST}:${REMOTE_DIR} към ${LOCAL_DIR}"

# 3. Използване на rsync за синхронизация
# Опции на rsync:
# -a: (archive mode) - рекурсивно копиране, запазва собственик, група, права, времена, символични връзки.
# -z: (compress) - компресира данни по време на трансфера, което пести трафик.
# -h: (human-readable) - показва размерите на файловете в по-четим формат (напр. KB, MB, GB).
# --progress: показва подробен прогрес на всеки файл.
# --include='vzdump-lxc-*.tar.zst': включва само файлове, които съвпадат с този шаблон.
# --exclude='*': изключва всички останали файлове, които не са изрично включени.
rsync_output=$(rsync -azh --progress \
    --include='vzdump-lxc-*.tar.zst' \
    --exclude='*' \
    "${REMOTE_HOST}:${REMOTE_DIR}/" "$LOCAL_DIR/" 2>&1)

# 4. Проверка на резултата от rsync
if [ $? -eq 0 ]; then
    echo "✅ Синхронизацията приключи успешно!"
else
    echo "❌ Грешка при синхронизацията с rsync."
    echo "Подробности за грешката:"
    echo "$rsync_output"
    exit 1
fi

# --- Функционалност за почистване на стари архиви ---
echo "🧹 Започвам почистване на стари архиви в $LOCAL_DIR..."

# 5. Изтриване на архиви по-стари от 2 дни
echo "🗑️ Изтривам архиви, по-стари от 2 дни..."
# Намира файлове, които са модифицирани преди повече от 2 дни (2 * 24 часа)
find "$LOCAL_DIR" -maxdepth 1 -type f -name 'vzdump-lxc-*.tar.zst' -mtime +2 -print0 | while IFS= read -r -d $'\0' old_file; do
    echo "   Изтривам по-стар от 2 дни: $(basename "$old_file")"
    rm -f "$old_file"
done

# 6. Изтриване на най-старите архиви, ако има повече от 2
echo "🗑️ Проверявам за излишни архиви (оставям само 2 най-нови)..."
# Изброява всички останали архиви, сортирани по дата на модификация (най-новите първи)
mapfile -t all_backups < <(find "$LOCAL_DIR" -maxdepth 1 -type f -name 'vzdump-lxc-*.tar.zst' -print0 | xargs -0 ls -t 2>/dev/null)

num_backups=${#all_backups[@]}

if [ "$num_backups" -gt 2 ]; then
    echo "   Намерени са $num_backups архива. Изтривам най-старите, за да останат само 2."
    # Взима всички файлове освен първите два (най-новите)
    files_to_delete=("${all_backups[@]:2}")
    
    for file_to_delete in "${files_to_delete[@]}"; do
        echo "   Изтривам излишен архив: $(basename "$file_to_delete")"
        rm -f "$file_to_delete"
    done
else
    echo "   Намерени са $num_backups архива. Няма излишни за изтриване (лимитът е 2)."
fi

echo "✅ Почистването приключи."
echo "✅ Готово! Всички архиви са в $LOCAL_DIR"
echo "Текущи архиви в $LOCAL_DIR:"
ls -lh "$LOCAL_DIR"/vzdump-lxc-*.tar.zst 2>/dev/null || echo "   Няма намерени архиви."