#!/bin/bash

# Proxmox VE Post-Install Configuration Script
# Автор: Федя Серафиев и Proxmox Helper Scripts
# Версия: 2.0 - Разширена версия

# Цветови променливи
RD=$(echo "\033[01;31m")
YW=$(echo "\033[33m")
GN=$(echo "\033[1;92m")
BL=$(echo "\033[36m")
MG=$(echo "\033[35m")
CL=$(echo "\033[m")
BFR="\\r\\033[K"
HOLD="-"
CM="${GN}✓${CL}"
CROSS="${RD}✗${CL}"
WARN="${YW}⚠${CL}"

# Лог файл
LOG_FILE="/var/log/proxmox-postinstall.log"

# Функции за съобщения
msg_info() {
    local msg="$1"
    echo -ne " ${HOLD} ${YW}${msg}..."
    echo "[INFO] $(date '+%Y-%m-%d %H:%M:%S') - ${msg}" >> "$LOG_FILE"
}

msg_ok() {
    local msg="$1"
    echo -e "${BFR} ${CM} ${GN}${msg}${CL}"
    echo "[OK] $(date '+%Y-%m-%d %H:%M:%S') - ${msg}" >> "$LOG_FILE"
}

msg_error() {
    local msg="$1"
    echo -e "${BFR} ${CROSS} ${RD}${msg}${CL}"
    echo "[ERROR] $(date '+%Y-%m-%d %H:%M:%S') - ${msg}" >> "$LOG_FILE"
}

msg_warn() {
    local msg="$1"
    echo -e "${BFR} ${WARN} ${YW}${msg}${CL}"
    echo "[WARN] $(date '+%Y-%m-%d %H:%M:%S') - ${msg}" >> "$LOG_FILE"
}

# Хедър информация
header_info() {
    clear
    cat <<"EOF"
    ____                                     _    ____________
   / __ \_________  _  ______ ___  ____  | |  / / ____/    
  / /_/ / ___/ __ \| |/_/ __ `__ \/ __ \ | | / / __/   
 / ____/ /  / /_/ />  </ / / / / / /_/ / | |/ / /___   
/_/   /_/   \____/_/|_/_/ /_/ /_/\____/  |___/_____/   
                                                        
    ____             __     ____           __        ____
   / __ \____  _____/ /_   /  _/___  _____/ /_____ _/ / /
  / /_/ / __ \/ ___/ __/   / // __ \/ ___/ __/ __ `/ / /
 / ____/ /_/ (__  ) /_   _/ // / / (__  ) /_/ /_/ / / / 
/_/    \____/____/\__/  /___/_/ /_/____/\__/\__,_/_/_/  
                                                        
           Конфигурационен скрипт за Proxmox VE
                    Версия 2.0 - Enhanced
EOF
    echo -e "${BL}════════════════════════════════════════════════════════════${CL}"
    echo -e "${GN}Добре дошли в разширения помощник за Proxmox VE!${CL}"
    echo -e "${BL}════════════════════════════════════════════════════════════${CL}"
    echo
}

# Проверка за root права
check_root() {
    if [[ $EUID -ne 0 ]]; then
        msg_error "Този скрипт трябва да бъде стартиран като root потребител"
        exit 1
    fi
}

# Проверка за Proxmox VE
check_proxmox() {
    if [[ ! -f /etc/pve/local/pve-ssl.pem ]]; then
        msg_error "Proxmox VE не е открит в тази система"
        exit 1
    fi
}

# Създаване на backup преди промени
create_backup() {
    local file="$1"
    if [[ -f "$file" ]]; then
        cp "$file" "${file}.backup-$(date +%Y%m%d-%H%M%S)"
        msg_ok "Създаден backup на $file"
    fi
}

# Функция за изключване на напомнянето за абонамент
disable_subscription_nag() {
    if [[ ! -f /etc/apt/apt.conf.d/no-nag-script ]]; then
        CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "НАПОМНЯНЕ ЗА АБОНАМЕНТ" --menu "Това ще изключи съобщението, което ви напомня да закупите абонамент всеки път, когато влезете в уеб интерфейса.\n \nИзключване на напомнянето за абонамент?" 14 58 2 \
          "yes" " " \
          "no" " " 3>&2 2>&1 1>&3)
        case $CHOICE in
        yes)
            whiptail --backtitle "Proxmox VE Helper Scripts" --msgbox --title "Поддръжка на абонаменти" "Подкрепата на екипа за разработка на софтуера е от съществено значение. Проверете официалния уебсайт за Поддръжка на абонаменти за цени. Без тяхната посветена работа нямаше да имаме този изключителен софтуер." 10 58
            msg_info "Изключване на напомнянето за абонамент"
            echo "DPkg::Post-Invoke { \"dpkg -V proxmox-widget-toolkit | grep -q '/proxmoxlib\.js$'; if [ \$? -eq 1 ]; then { echo 'Премахване на напомнянето от интерфейса...'; sed -i '/.*data\.status.*{/{s/\!//;s/active/NoMoreNagging/}' /usr/share/javascript/proxmox-widget-toolkit/proxmoxlib.js; }; fi\"; };" >/etc/apt/apt.conf.d/no-nag-script
            apt --reinstall install proxmox-widget-toolkit &>/dev/null
            msg_ok "Напомнянето за абонамент е изключено (Изтрийте кеша на браузъра)"
            ;;
        no)
            msg_error "Избрахте 'не' за изключване на напомнянето"
            ;;
        esac
    else
        msg_ok "Напомнянето за абонамент вече е изключено"
    fi
}

# Функция за корекция на източниците
fix_sources() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "КОРЕКЦИЯ НА ИЗТОЧНИЦИТЕ" --menu "Това ще коригира източниците на Proxmox VE за правилна работа.\n \nКорекция на източниците на Proxmox VE?" 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Коригиране на източниците на Proxmox VE"
        create_backup "/etc/apt/sources.list"
        cat <<EOF >/etc/apt/sources.list
deb http://deb.debian.org/debian bookworm main contrib
deb http://deb.debian.org/debian bookworm-updates main contrib
deb http://security.debian.org/debian-security bookworm-security main contrib
EOF
        echo 'APT::Get::Update::SourceListWarnings::NonFreeFirmware "false";' >/etc/apt/apt.conf.d/no-bookworm-firmware.conf
        msg_ok "Източниците на Proxmox VE са коригирани"
        ;;
    no)
        msg_error "Избрахте 'не' за корекция на източниците"
        ;;
    esac
}

# Функция за изключване на enterprise хранилище
disable_enterprise_repo() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "PVE-ENTERPRISE" --menu "Хранилището 'pve-enterprise' е достъпно само за потребители, закупили абонамент за Proxmox VE.\n \nИзключване на 'pve-enterprise' хранилище?" 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Изключване на 'pve-enterprise' хранилище"
        create_backup "/etc/apt/sources.list.d/pve-enterprise.list"
        cat <<EOF >/etc/apt/sources.list.d/pve-enterprise.list
# deb https://enterprise.proxmox.com/debian/pve bookworm pve-enterprise
EOF
        msg_ok "'pve-enterprise' хранилището е изключено"
        ;;
    no)
        msg_error "Избрахте 'не' за изключване на хранилището"
        ;;
    esac
}

# Функция за активиране на no-subscription хранилище
enable_no_subscription_repo() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "PVE-NO-SUBSCRIPTION" --menu "Хранилището 'pve-no-subscription' предоставя достъп до всички компоненти с отворен код на Proxmox VE.\n \nАктивиране на 'pve-no-subscription' хранилище?" 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Активиране на 'pve-no-subscription' хранилище"
        cat <<EOF >/etc/apt/sources.list.d/pve-install-repo.list
deb http://download.proxmox.com/debian/pve bookworm pve-no-subscription
EOF
        msg_ok "'pve-no-subscription' хранилището е активирано"
        ;;
    no)
        msg_error "Избрахте 'не' за активиране на хранилището"
        ;;
    esac
}

# Функция за обновяване на системата
update_system() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "ОБНОВЯВАНЕ НА СИСТЕМАТА" --menu "Това ще обнови всички пакети в системата.\n \nОбновяване на системата?" 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Обновяване на списъка с пакети"
        apt update &>/dev/null
        msg_ok "Списъкът с пакети е обновен"
        
        msg_info "Обновяване на системата"
        apt full-upgrade -y &>/dev/null
        msg_ok "Системата е обновена"
        
        msg_info "Почистване на ненужни пакети"
        apt autoremove -y &>/dev/null
        apt autoclean &>/dev/null
        msg_ok "Ненужните пакети са премахнати"
        ;;
    no)
        msg_error "Избрахте 'не' за обновяване на системата"
        ;;
    esac
}

# Функция за конфигуриране на firewall
configure_firewall() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "FIREWALL КОНФИГУРАЦИЯ" --menu "Активиране на основна firewall конфигурация?\n \nТова ще активира основни firewall правила за защита." 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Конфигуриране на firewall"
        pvesh set /cluster/firewall/options --enable 1 &>/dev/null
        msg_ok "Firewall е активиран"
        ;;
    no)
        msg_error "Избрахте 'не' за активиране на firewall"
        ;;
    esac
}

# НОВА ФУНКЦИЯ: Инсталиране на полезни пакети
install_useful_packages() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "ИНСТАЛАЦИЯ НА ПАКЕТИ" --checklist "Изберете пакети за инсталиране:" 20 70 10 \
      "htop" "Интерактивен мониторинг на процеси" OFF \
      "iotop" "Мониторинг на I/O операции" OFF \
      "ncdu" "Анализатор на дисково пространство" OFF \
      "vim" "Разширен текстов редактор" OFF \
      "net-tools" "Мрежови инструменти" OFF \
      "curl" "Инструмент за трансфер на данни" OFF \
      "wget" "Инструмент за изтегляне" OFF \
      "git" "Система за контрол на версии" OFF \
      "tmux" "Терминален мултиплексор" OFF \
      "screen" "Терминален мултиплексор" OFF 3>&2 2>&1 1>&3)
    
    if [[ -n "$CHOICE" ]]; then
        msg_info "Инсталиране на избраните пакети"
        PACKAGES=$(echo "$CHOICE" | tr -d '"')
        apt update &>/dev/null
        apt install -y $PACKAGES &>/dev/null
        msg_ok "Пакетите са инсталирани успешно"
    else
        msg_error "Не са избрани пакети за инсталиране"
    fi
}

# НОВА ФУНКЦИЯ: Оптимизация на swap
optimize_swap() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "SWAP ОПТИМИЗАЦИЯ" --menu "Оптимизиране на swap настройките?\n \nТова ще подобри производителността на системата." 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Оптимизиране на swap настройките"
        
        # Проверка текущи настройки
        current_swappiness=$(cat /proc/sys/vm/swappiness)
        current_cache_pressure=$(cat /proc/sys/vm/vfs_cache_pressure)
        
        # Задаване на нови стойности
        echo "vm.swappiness=10" >> /etc/sysctl.conf
        echo "vm.vfs_cache_pressure=50" >> /etc/sysctl.conf
        sysctl -p &>/dev/null
        
        msg_ok "Swap е оптимизиран (swappiness: $current_swappiness -> 10, cache_pressure: $current_cache_pressure -> 50)"
        ;;
    no)
        msg_error "Избрахте 'не' за оптимизация на swap"
        ;;
    esac
}

# НОВА ФУНКЦИЯ: Конфигуриране на автоматични backups
configure_auto_backup() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "АВТОМАТИЧНИ BACKUPS" --menu "Конфигуриране на автоматични backups?\n \nТова ще настрои планиран backup на VMs/CTs." 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        BACKUP_TIME=$(whiptail --backtitle "Proxmox VE Helper Scripts" --inputbox "Въведете час за backup (0-23):" 10 60 "2" 3>&1 1>&2 2>&3)
        BACKUP_STORAGE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --inputbox "Въведете име на storage за backups:" 10 60 "local" 3>&1 1>&2 2>&3)
        BACKUP_RETENTION=$(whiptail --backtitle "Proxmox VE Helper Scripts" --inputbox "Брой backups за запазване:" 10 60 "3" 3>&1 1>&2 2>&3)
        
        msg_info "Конфигуриране на автоматични backups"
        
        # Създаване на backup задача
        pvesh create /cluster/backup --schedule "0 $BACKUP_TIME * * *" \
          --storage "$BACKUP_STORAGE" \
          --mode snapshot \
          --compress zstd \
          --prune-backups "keep-last=$BACKUP_RETENTION" \
          --all 1 \
          --enabled 1 &>/dev/null
        
        msg_ok "Автоматичните backups са конфигурирани (час: $BACKUP_TIME, storage: $BACKUP_STORAGE, запазване: $BACKUP_RETENTION)"
        ;;
    no)
        msg_error "Избрахте 'не' за конфигуриране на backups"
        ;;
    esac
}

# НОВА ФУНКЦИЯ: Настройване на email известия
configure_email_notifications() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "EMAIL ИЗВЕСТИЯ" --menu "Конфигуриране на email известия?\n \nТова ще настрои изпращане на email при проблеми." 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        EMAIL=$(whiptail --backtitle "Proxmox VE Helper Scripts" --inputbox "Въведете вашия email адрес:" 10 60 3>&1 1>&2 2>&3)
        
        if [[ -n "$EMAIL" ]]; then
            msg_info "Конфигуриране на email известия"
            
            # Инсталиране на postfix ако не е инсталиран
            if ! command -v postfix &> /dev/null; then
                DEBIAN_FRONTEND=noninteractive apt install -y postfix mailutils &>/dev/null
            fi
            
            # Настройка на root email
            echo "root: $EMAIL" >> /etc/aliases
            newaliases &>/dev/null
            
            msg_ok "Email известията са конфигурирани за $EMAIL"
        else
            msg_error "Невалиден email адрес"
        fi
        ;;
    no)
        msg_error "Избрахте 'не' за конфигуриране на email"
        ;;
    esac
}

# НОВА ФУНКЦИЯ: Оптимизация на ZFS (ако е налично)
optimize_zfs() {
    if ! command -v zfs &> /dev/null; then
        msg_warn "ZFS не е открит в системата"
        return
    fi
    
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "ZFS ОПТИМИЗАЦИЯ" --menu "Оптимизиране на ZFS настройките?\n \nТова ще подобри производителността на ZFS." 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Оптимизиране на ZFS настройките"
        
        # Ограничаване на ARC памет (50% от RAM)
        TOTAL_MEM=$(free -b | grep '^Mem:' | awk '{print $2}')
        ARC_MAX=$((TOTAL_MEM / 2))
        
        echo "options zfs zfs_arc_max=$ARC_MAX" > /etc/modprobe.d/zfs.conf
        
        msg_ok "ZFS е оптимизиран (ARC max: $(($ARC_MAX / 1024 / 1024 / 1024))GB)"
        msg_warn "Необходим е рестарт за прилагане на промените"
        ;;
    no)
        msg_error "Избрахте 'не' за оптимизация на ZFS"
        ;;
    esac
}

# НОВА ФУНКЦИЯ: Почистване на логове
cleanup_logs() {
    CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "ПОЧИСТВАНЕ НА ЛОГОВЕ" --menu "Почистване на стари лог файлове?\n \nТова ще освободи дисково пространство." 14 58 2 \
      "yes" " " \
      "no" " " 3>&2 2>&1 1>&3)
    case $CHOICE in
    yes)
        msg_info "Почистване на стари логове"
        
        # Почистване на журнални логове (запазване на последните 7 дни)
        journalctl --vacuum-time=7d &>/dev/null
        
        # Почистване на стари apt кешове
        apt clean &>/dev/null
        
        # Почистване на стари ротирани логове
        find /var/log -type f -name "*.gz" -mtime +30 -delete 2>/dev/null
        find /var/log -type f -name "*.1" -mtime +7 -delete 2>/dev/null
        
        msg_ok "Старите логове са почистени"
        ;;
    no)
        msg_error "Избрахте 'не' за почистване на логове"
        ;;
    esac
}

# НОВА ФУНКЦИЯ: Преглед на лог файла
view_log() {
    if [[ -f "$LOG_FILE" ]]; then
        whiptail --backtitle "Proxmox VE Helper Scripts" --title "ЛОГ ФАЙЛ" --scrolltext --textbox "$LOG_FILE" 20 70
    else
        whiptail --backtitle "Proxmox VE Helper Scripts" --msgbox "Лог файлът не съществува" 8 40
    fi
}

# ПОДОБРЕНА ФУНКЦИЯ: Показване на системна информация
show_system_info() {
    clear
    header_info
    echo -e "${BL}════════════════════════════════════════════════════════════${CL}"
    echo -e "${GN}                    СИСТЕМНА ИНФОРМАЦИЯ                     ${CL}"
    echo -e "${BL}════════════════════════════════════════════════════════════${CL}"
    echo
    echo -e "${YW}Proxmox VE версия:${CL} $(pveversion | head -1)"
    echo -e "${YW}Операционна система:${CL} $(lsb_release -d | cut -f2)"
    echo -e "${YW}Kernel версия:${CL} $(uname -r)"
    echo -e "${YW}Архитектура:${CL} $(uname -m)"
    echo -e "${YW}Хостнейм:${CL} $(hostname)"
    echo -e "${YW}Възходящо време:${CL} $(uptime -p)"
    echo -e "${YW}Натоварване:${CL} $(uptime | awk -F'load average:' '{print $2}')"
    echo
    echo -e "${MG}=== РЕСУРСИ ===${CL}"
    echo -e "${YW}CPU модел:${CL} $(lscpu | grep 'Model name' | cut -d: -f2 | xargs)"
    echo -e "${YW}CPU ядра:${CL} $(nproc)"
    echo -e "${YW}Памет:${CL} $(free -h | grep '^Mem:' | awk '{print $3"/"$2" ("$3/$2*100"%)"}')"
    echo -e "${YW}Swap:${CL} $(free -h | grep '^Swap:' | awk '{print $3"/"$2}')"
    echo
    echo -e "${MG}=== ДИСКОВЕ ===${CL}"
    df -h | grep -E '^/dev/' | awk '{printf "%-20s %10s %10s %10s %s\n", $1, $2, $3, $4, $5}'
    echo
    echo -e "${MG}=== МРЕЖА ===${CL}"
    ip -4 addr show | grep -oP '(?<=inet\s)\d+(\.\d+){3}' | grep -v '127.0.0.1' | head -5
    echo
    echo -e "${BL}════════════════════════════════════════════════════════════${CL}"
    echo
    read -p "Натиснете Enter за връщане в главното меню..."
}

# ПОДОБРЕНО: Главно меню с повече опции
main_menu() {
    while true; do
        header_info
        CHOICE=$(whiptail --backtitle "Proxmox VE Helper Scripts v2.0" --title "ГЛАВНО МЕНЮ" --menu "Изберете опция:" 24 78 16 \
          "1" "Изключване на напомнянето за абонамент" \
          "2" "Корекция на източниците" \
          "3" "Изключване на enterprise хранилище" \
          "4" "Активиране на no-subscription хранилище" \
          "5" "Обновяване на системата" \
          "6" "Конфигуриране на firewall" \
          "7" "Инсталиране на полезни пакети" \
          "8" "Оптимизация на swap" \
          "9" "Конфигуриране на автоматични backups" \
          "10" "Настройване на email известия" \
          "11" "Оптимизация на ZFS" \
          "12" "Почистване на логове" \
          "13" "Преглед на лог файла" \
          "14" "Показване на системна информация" \
          "15" "Изпълнение на всички конфигурации" \
          "16" "Изход" 3>&2 2>&1 1>&3)
        
        case $CHOICE in
        1) disable_subscription_nag ;;
        2) fix_sources ;;
        3) disable_enterprise_repo ;;
        4) enable_no_subscription_repo ;;
        5) update_system ;;
        6) configure_firewall ;;
        7) install_useful_packages ;;
        8) optimize_swap ;;
        9) configure_auto_backup ;;
        10) configure_email_notifications ;;
        11) optimize_zfs ;;
        12) cleanup_logs ;;
        13) view_log ;;
        14) show_system_info ;;
        15)
            echo -e "${GN}Изпълнение на всички основни конфигурации...${CL}"
            echo
            disable_subscription_nag
            fix_sources
            disable_enterprise_repo
            enable_no_subscription_repo
            update_system
            configure_firewall
            optimize_swap
            cleanup_logs
            echo
            echo -e "${GN}Всички конфигурации са завършени!${CL}"
            echo -e "${YW}Препоръчва се рестартиране на системата.${CL}"
            echo
            REBOOT=$(whiptail --backtitle "Proxmox VE Helper Scripts" --title "РЕСТАРТИРАНЕ" --yesno "Искате ли да рестартирате системата сега?" 10 60 3>&1 1>&2 2>&3)
            if [[ $? -eq 0 ]]; then
                msg_info "Рестартиране на системата след 5 секунди..."
                sleep 5
                reboot
            fi
            ;;
        16)
            echo -e "${GN}Благодарим ви за използването на Proxmox VE Helper Scripts!${CL}"
            exit 0
            ;;
        *)
            msg_error "Невалидна опция"
            ;;
        esac
        
        echo
        read -p "Натиснете Enter за връщане в главното меню..."
    done
}

# Основна функция
main() {
    check_root
    check_proxmox
    
    # Създаване на лог файл ако не съществува
    touch "$LOG_FILE"
    echo "=== Proxmox Post-Install Script Log ===" >> "$LOG_FILE"
    echo "Стартиран на: $(date)" >> "$LOG_FILE"
    
    # Проверка за whiptail
    if ! command -v whiptail &> /dev/null; then
        msg_info "Инсталиране на whiptail"
        apt update &>/dev/null
        apt install -y whiptail &>/dev/null
        msg_ok "whiptail е инсталиран"
    fi
    
    main_menu
}

# Стартиране на скрипта
main "$@"