# ===================================================================
# PROFESSIONAL VS CODE + PYTHON + GIT + DOCKER SETUP FOR WINDOWS 11
# Run as Administrator | UTF-8 without BOM
# ===================================================================

$isAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
if (-not $isAdmin) {
    Write-Host "ERROR: Run as Administrator!" -ForegroundColor Red
    pause
    exit 1
}

Write-Host "`nStarting professional dev environment setup..." -ForegroundColor Green

# --- 1. Ensure winget is available (Robust) ---
function Install-Winget {
    if (Get-Command winget -ErrorAction SilentlyContinue) { return $true }
    
    Write-Host "winget not found. Installing App Installer..." -ForegroundColor Cyan
    try {
        # Try Microsoft Store (modern)
        Add-AppxPackage -RegisterByFamilyName -MainPackage Microsoft.DesktopAppInstaller_8wekyb3d8bbwe -ErrorAction SilentlyContinue
        Start-Sleep -Seconds 3
        if (Get-Command winget -ErrorAction SilentlyContinue) { return $true }
    } catch {}

    # Fallback: Download .msixbundle from GitHub
    Write-Host "Downloading App Installer from GitHub..." -ForegroundColor Yellow
    $url = "https://aka.ms/getwinget"
    $out = "$env:TEMP\winget.msixbundle"
    Invoke-WebRequest -Uri $url -OutFile $out -UseBasicParsing
    Add-AppxPackage -Path $out -ErrorAction SilentlyContinue
    Remove-Item $out -Force
    Start-Sleep -Seconds 5
    return (Get-Command winget -ErrorAction SilentlyContinue)
}

if (-not (Install-Winget)) {
    Write-Host "FAILED to install winget. Install manually: https://aka.ms/getwinget" -ForegroundColor Red
    pause
    exit 1
}

# --- 2. Install VS Code ---
Write-Host "`nInstalling VS Code..." -ForegroundColor Cyan
if (-not (Get-Command code -ErrorAction SilentlyContinue)) {
    winget install -e --id Microsoft.VisualStudioCode --source winget --silent --accept-package-agreements --accept-source-agreements
    Start-Sleep -Seconds 10  # Wait for PATH update
    $env:PATH = [System.Environment]::GetEnvironmentVariable("PATH","Machine") + ";" + [System.Environment]::GetEnvironmentVariable("PATH","User")
}
if (Get-Command code -ErrorAction SilentlyContinue) {
    Write-Host "VS Code ready." -ForegroundColor Green
} else {
    Write-Host "VS Code install failed." -ForegroundColor Red
    pause
    exit 1
}

# --- 3. Install Python + pip upgrade ---
Write-Host "`nInstalling Python 3.12..." -ForegroundColor Cyan
if (-not (Get-Command python -ErrorAction SilentlyContinue)) {
    winget install -e --id Python.Python.3.12 --source winget --silent
    Start-Sleep -Seconds 10
}
if (Get-Command python -ErrorAction SilentlyContinue) {
    Write-Host "Upgrading pip..." -ForegroundColor Cyan
    python -m pip install --upgrade pip --quiet
    Write-Host "Python ready." -ForegroundColor Green
}

# --- 4. Install Git ---
Write-Host "`nInstalling Git..." -ForegroundColor Cyan
if (-not (Get-Command git -ErrorAction SilentlyContinue)) {
    winget install -e --id Git.Git --source winget --silent
}
if (Get-Command git -ErrorAction SilentlyContinue) {
    Write-Host "Git ready." -ForegroundColor Green
}

# --- 5. Install Docker Desktop ---
Write-Host "`nInstalling Docker Desktop..." -ForegroundColor Cyan
if (-not (Get-Command docker -ErrorAction SilentlyContinue)) {
    winget install -e --id Docker.DockerDesktop --source winget --silent --accept-package-agreements
    Write-Host "Docker installed. Restart may be required." -ForegroundColor Yellow
}

# --- 6. Safe settings.json merge ---
$settingsPath = "$env:APPDATA\Code\User\settings.json"
$settingsDir = Split-Path $settingsPath -Parent
if (-not (Test-Path $settingsDir)) { New-Item -ItemType Directory -Path $settingsDir -Force | Out-Null }

$defaultSettings = @{
    "editor.fontSize" = 14
    "editor.tabSize" = 4
    "editor.insertSpaces" = $true
    "files.autoSave" = "afterDelay"
    "files.autoSaveDelay" = 1000
    "workbench.colorTheme" = "Default Dark+"
    "terminal.integrated.defaultProfile.windows" = "PowerShell"
    "python.defaultInterpreterPath" = "python"
    "python.terminal.activateEnvironment" = $true
    "git.autofetch" = $true
    "git.confirmSync" = $false
    "extensions.autoUpdate" = $true
    "telemetry.telemetryLevel" = "off"
    "workbench preferred language" = "bg"  # Bulgarian
}

# Load existing or create new
if (Test-Path $settingsPath) {
    try {
        $current = Get-Content $settingsPath -Raw | ConvertFrom-Json -ErrorAction Stop
        if ($current -isnot [PSCustomObject]) { $current = [PSCustomObject]@{} }
    } catch {
        Write-Host "Invalid settings.json - backing up and replacing..." -ForegroundColor Yellow
        Copy-Item $settingsPath "$settingsPath.bak" -Force
        $current = [PSCustomObject]@{}
    }
} else {
    $current = [PSCustomObject]@{}
}

# Merge settings
foreach ($key in $defaultSettings.Keys) {
    if (-not ($current.PSObject.Properties.Name -contains $key)) {
        Add-Member -InputObject $current -NotePropertyName $key -NotePropertyValue $defaultSettings[$key]
    } else {
        $current.$key = $defaultSettings[$key]
    }
}

$json = $current | ConvertTo-Json -Depth 10
[System.IO.File]::WriteAllText($settingsPath, $json, [System.Text.Encoding]::UTF8)
Write-Host "settings.json updated safely!" -ForegroundColor Green

# --- 7. Install Extensions ---
Write-Host "`nInstalling VS Code extensions..." -ForegroundColor Cyan
$extensions = @(
    "ms-python.python"
    "ms-python.vscode-pylance"
    "ms-toolsai.jupyter"
    "eamodio.gitlens"
    "GitHub.copilot"
    "ms-vscode.bulgarian-language-pack"
    "streetsidesoftware.code-spell-checker"
    "esbenp.prettier-vscode"
    "ms-vscode-remote.remote-wsl"
    "ms-azuretools.vscode-docker"
)

foreach ($ext in $extensions) {
    if (-not (code --list-extensions | Where-Object { $_ -eq $ext })) {
        Write-Host "  Installing: $ext"
        code --install-extension $ext --force | Out-Null
    } else {
        Write-Host "  Already: $ext" -ForegroundColor DarkGray
    }
}

# --- 8. Activate Bulgarian Language ---
Write-Host "`nActivating Bulgarian interface..." -ForegroundColor Cyan
$localePath = "$env:APPDATA\Code\User\locale.json"
$localeDir = Split-Path $localePath -Parent
if (-not (Test-Path $localeDir)) { New-Item -ItemType Directory -Path $localeDir -Force | Out-Null }
'{"locale":"bg"}' | Out-File -FilePath $localePath -Encoding UTF8 -Force
Write-Host "Bulgarian language activated! Restart VS Code." -ForegroundColor Green

# --- 9. WSL2 Check ---
Write-Host "`nChecking WSL2..." -ForegroundColor Cyan
if (Get-Command wsl -ErrorAction SilentlyContinue) {
    $distros = wsl --list --quiet | Where-Object { $_ -and $_.Trim() }
    if ($distros) {
        Write-Host "WSL2 active: $distros" -ForegroundColor Green
    } else {
        $ans = Read-Host "Install Ubuntu in WSL2? (y/n)"
        if ($ans -match "^[Yy]") {
            wsl --install -d Ubuntu
            Write-Host "Restart required after install!" -ForegroundColor Yellow
        }
    }
} else {
    $ans = Read-Host "Enable WSL2? (y/n)"
    if ($ans -match "^[Yy]") {
        wsl --install
        Write-Host "Restart required!" -ForegroundColor Yellow
    }
}

# --- Final ---
Write-Host "`nSETUP COMPLETE!" -ForegroundColor Green
Write-Host "   VS Code: code ." -ForegroundColor Cyan
Write-Host "   Python: python --version" -ForegroundColor Cyan
Write-Host "   Docker: docker --version" -ForegroundColor Cyan
Write-Host "   Restart VS Code to see Bulgarian UI!" -ForegroundColor Magenta
Write-Host "`nTip: Restart PC if Docker/WSL/Python was installed.`n"

pause