#!/bin/bash
clear

echo -e "\e[1;36m"
cat << "EOF"
  ███████╗███████╗██████╗ ██╗   ██╗ █████╗     ██╗   ██╗██████╗ ██╗  ████████╗██████╗  █████╗ 
  ██╔════╝██╔════╝██╔══██╗╚██╗ ██╔╝██╔══██╗    ██║   ██║██╔══██╗██║  ╚══██╔══╝██╔══██╗██╔══██╗
  █████╗  █████╗  ██║  ██║ ╚████╔╝ ███████║    ██║   ██║██████╔╝██║     ██║   ██████╔╝███████║
  ██╔══╝  ██╔══╝  ██║  ██║  ╚██╔╝  ██╔══██║    ██║   ██║██╔══██╗██║     ██║   ██╔══██╗██╔══██║
  ██║     ███████╗██████╔╝   ██║   ██║  ██║    ╚██████╔╝██║  ██║███████╗██║   ██║  ██║██║  ██║
  ╚═╝     ╚══════╝╚═════╝    ╚═╝   ╚═╝  ╚═╝     ╚═════╝ ╚═╝  ╚═╝╚══════╝╚═╝   ╚═╝  ╚═╝╚═╝  ╚═╝
EOF
echo -e "\e[0m"

echo -e "\e[1;35mFEDYA PRO FILE SERVER v5.0 — PROFESSIONAL EDITION\e[0m"
echo -e "\e[1;34m══════════════════════════════════════════════════════════════════════\e[0m"
echo

# Проверка за root права
[ "$EUID" -ne 0 ] && { echo -e "\e[1;31m❌ sudo required!\e[0m"; exit 1; }

# Събиране на потребителски настройки
echo -e "\e[1;33m⚙️  Server Configuration:\e[0m"
read -p "$(echo -e '\e[1;36m📁 Files directory\e[0m [\e[1;32m/var/www/files\e[0m]: ')" DIR
DIR=${DIR:-"/var/www/files"}
mkdir -p "$DIR"

read -p "$(echo -e '\e[1;36m🌐 Port\e[0m [\e[1;32m8080\e[0m]: ')" PORT
PORT=${PORT:-"8080"}

read -p "$(echo -e '\e[1;36m🏷️  Server name\e[0m [\e[1;32mFedya'\''s File Server\e[0m]: ')" NAME
NAME=${NAME:-"Fedya's File Server"}

read -p "$(echo -e '\e[1;36m🔧 Install as systemd service?\e[0m [\e[1;32mY\e[0m/n]: ')" SVC
[[ "$SVC" =~ ^[Nn]$ ]] && SERVICE=false || SERVICE=true

echo -e "\n\e[1;33m📦 Installing Go...\e[0m"
apt update -qq && apt install -y golang-go &>/dev/null || true

# Подготовка на временна директория
TMP=$(mktemp -d)
cd "$TMP"

echo -e "\e[1;33m🔨 Creating Professional Go Server...\e[0m"

# Създаване на PROFESSIONAL Go код с всички действия (FIXED VERSION)
cat > main.go <<'EOF'
package main

import (
	"archive/zip"
	"crypto/rand"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"html/template"
	"image"
	"image/jpeg"
	_ "image/gif"
	_ "image/png"
	"io"
	"log"
	"net/http"
	"os"
	"path"
	"path/filepath"
	"sort"
	"strings"
	"time"
)

var (
	fileDir string
	absFileDir string
	port string
	serverName string
	tmpl *template.Template
	links = make(map[string]struct {
		Path string
		Expiry time.Time
		OneTime bool
		Used bool
	})
)

func init() {
	fileDir = os.Getenv("FILE_DIR")
	if fileDir == "" {
		fileDir = "/var/www/files"
	}
	port = os.Getenv("PORT")
	if port == "" {
		port = "8080"
	}
	serverName = os.Getenv("SERVER_NAME")
	if serverName == "" {
		serverName = "Fedya's File Server"
	}
    
	var err error
	fileDir, err = filepath.Abs(fileDir)
	if err != nil {
		log.Fatalf("Error getting absolute path: %v", err)
	}
	absFileDir = fileDir
	
	// Create base directories
	os.MkdirAll(filepath.Join(fileDir, "pdf"), 0755)
	os.MkdirAll(filepath.Join(fileDir, "images"), 0755)
    
	tmpl = template.Must(template.New("index").Parse(html))
}

const html = `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>{{.Name}}</title>
    <!-- PROFESSIONAL FAVICON -->
    <link rel="icon" type="image/svg+xml" href="data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNjQiIGhlaWdodD0iNjQiIHZpZXdCb3g9IjAgMCA2NCA2NCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICA8cmVjdCB3aWR0aD0iNjQiIGhlaWdodD0iNjQiIHJ4PSIxMiIgZmlsbD0idXJsKCNncmFkKSIvPgogIDxkZWZzPgogICAgPGxpbmVhckdyYWRpZW50IGlkPSJncmFkIiB4MT0iMCUiIHkxPSIwJSIgeDI9IjEwMCUiIHkyPSIxMDAlIj4KICAgICAgPHN0b3Agb2Zmc2V0PSIwJSIgc3R5bGU9InN0b3AtY29sb3I6IzRGNjBFNTtzdG9wLW9wYWNpdHk6MSIvPgogICAgICA8c3RvcCBvZmZzZXQ9IjEwMCUiIHN0eWxlPSJzdG9wLWNvbG9yOiMwNkI2RDQ7c3RvcC1vcGFjaXR5OjEiLz4KICAgIDwvbGluZWFyR3JhZGllbnQ+CiAgPC9kZWZzPgogIDxwYXRoIGQ9Ik0yMCAyMEMyMCAxNy43OTA5IDE3Ljc5MDkgMTYgMTUgMTZDMTIuMjA5MSAxNiAxMCAxNy43OTA5IDEwIDIwQzEwIDIyLjIwOTEgMTIuMjA5MSAyNCAxNSAyNEMxNy43OTA5IDI0IDIwIDIyLjIwOTEgMjAgMjBaIiBmaWxsPSJ3aGl0ZSIvPgogIDxwYXRoIGQ9Ik0zOCAyMEMzOCAxNy43OTA5IDM1Ljc5MDkgMTYgMzMgMTZDMzAuMjA5MSAxNiAyOCAxNy43OTA5IDI4IDIwQzI4IDIyLjIwOTEgMzAuMjA5MSAyNCAzMyAyNEMzNS43OTA5IDI0IDM4IDIyLjIwOTEgMzggMjBaIiBmaWxsPSJ3aGl0ZSIvPgogIDxwYXRoIGQ9Ik01NCAyMEM1NCAxNy43OTA5IDUxLjc5MDkgMTYgNDkgMTZDNDYuMjA5MSAxNiA0NCAxNy43OTA5IDQ0IDIwQzQ0IDIyLjIwOTEgNDYuMjA5MSAyNCA0OSAyNEM1MS43OTA5IDI0IDU0IDIyLjIwOTEgNTQgMjBaIiBmaWxsPSJ3aGl0ZSIvPgogIDxwYXRoIGQ9Ik0yMCA0MEMyMCAzNy43OTA5IDE3Ljc5MDkgMzYgMTUgMzZDMTIuMjA5MSAzNiAxMCAzNy43OTA5IDEwIDQwQzEwIDQyLjIwOTEgMTIuMjA5MSA0NCAxNSA0NEMxNy43OTA5IDQ0IDIwIDQyLjIwOTEgMjAgNDBaIiBmaWxsPSJ3aGl0ZSIvPgogIDxwYXRoIGQ9Ik0zOCA0MEMzOCAzNy43OTA5IDM1Ljc5MDkgMzYgMzMgMzZDMzAuMjA5MSAzNiAyOCAzNy43OTA5IDI4IDQwQzI8IDQyLjIwOTEgMzAuMjA5MSA0NCAzMyA0NEMzNS43OTA5IDQ0IDM4IDQyLjIwOTEgMzggNDBaIiBmaWxsPSJ3aGl0ZSIvPgo8L3N2Zz4K"/>
    <style>
        :root {
            --primary: #4F46E5;
            --primary-dark: #4338CA;
            --secondary: #06B6D4;
            --success: #10B981;
            --danger: #EF4444;
            --warning: #F59E0B;
            --info: #8B5CF6;
            --dark: #1E293B;
            --light: #F8FAFC;
            --gray: #64748B;
            --border: #E2E8F0;
        }
        
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', system-ui, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: var(--dark);
            min-height: 100vh;
            padding: 20px;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1800px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            box-shadow: 0 25px 80px rgba(0,0,0,0.15);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 3rem 2rem;
            text-align: center;
            position: relative;
            overflow: hidden;
        }
        
        .header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url("data:image/svg+xml,%3Csvg width='100' height='100' viewBox='0 0 100 100' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M11 18c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm48 25c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm-43-7c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm63 31c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zM34 90c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm56-76c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zM12 86c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm28-65c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm23-11c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zm-6 60c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm29 22c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zM32 63c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zm57-13c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zm-9-21c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2zM60 91c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2zM35 41c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2zM12 60c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2z' fill='%23ffffff' fill-opacity='0.1' fill-rule='evenodd'/%3E%3C/svg%3E");
        }
        
        .logo {
            font-size: 3.5rem;
            font-weight: 900;
            background: linear-gradient(135deg, #FFF, #E0F2FE);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 0.5rem;
            text-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .subtitle {
            font-size: 1.3rem;
            opacity: 0.9;
            font-weight: 300;
        }
        
        .stats {
            padding: 1.5rem;
            background: var(--light);
            text-align: center;
            font-weight: 600;
            display: flex;
            justify-content: center;
            gap: 2rem;
            flex-wrap: wrap;
        }
        
        .stat-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--gray);
        }
        
        .path {
            padding: 1.2rem 2rem;
            background: var(--light);
            border-bottom: 1px solid var(--border);
            font-weight: 600;
            color: var(--primary);
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }
        
        .actions {
            padding: 2rem;
            background: var(--light);
            display: flex;
            flex-wrap: wrap;
            gap: 1.5rem;
            align-items: center;
            border-bottom: 1px solid var(--border);
        }
        
        .upload-box {
            background: white;
            padding: 2rem;
            border-radius: 16px;
            box-shadow: 0 8px 25px rgba(139,92,246,0.1);
            border: 2px dashed #DDD6FE;
            flex: 1;
            min-width: 360px;
            transition: all 0.3s ease;
        }
        
        .upload-box:hover {
            border-color: var(--primary);
            transform: translateY(-3px);
            box-shadow: 0 15px 35px rgba(139,92,246,0.15);
        }
        
        button, .btn {
            padding: 12px 24px;
            border: none;
            border-radius: 12px;
            color: white;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            text-decoration: none;
            font-size: 0.95rem;
        }
        
        button:hover, .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(79,70,229,0.3);
        }
        
        .btn-success {
            background: linear-gradient(135deg, var(--success), #059669);
        }
        
        .btn-warning {
            background: linear-gradient(135deg, var(--warning), #D97706);
        }
        
        .btn-danger {
            background: linear-gradient(135deg, var(--danger), #DC2626);
        }
        
        .btn-info {
            background: linear-gradient(135deg, var(--info), #7C3AED);
        }
        
        .action-btn {
            padding: 8px 12px;
            border: none;
            border-radius: 8px;
            color: white;
            cursor: pointer;
            transition: all 0.2s ease;
            display: inline-flex;
            align-items: center;
            gap: 0.3rem;
            font-size: 0.85rem;
            margin: 2px;
        }
        
        .action-btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        
        .btn-download { background: var(--success); }
        .btn-delete { background: var(--danger); }
        .btn-rename { background: var(--warning); }
        .btn-link { background: var(--info); }
        .btn-info { background: var(--primary); }
        .btn-zip { background: #7C3AED; }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.7);
            align-items: center;
            justify-content: center;
            z-index: 1000;
        }
        
        .modal-content {
            background: white;
            padding: 2.5rem;
            border-radius: 20px;
            max-width: 500px;
            width: 90%;
            box-shadow: 0 25px 50px rgba(0,0,0,0.25);
        }
        
        .close {
            float: right;
            font-size: 2rem;
            cursor: pointer;
            color: var(--gray);
        }
        
        table {
            width: 100%;
            border-collapse: collapse;
            background: white;
        }
        
        th {
            background: var(--light);
            padding: 1.5rem;
            text-align: left;
            font-weight: 700;
            color: var(--gray);
            border-bottom: 3px solid var(--border);
        }
        
        td {
            padding: 1.2rem 1.5rem;
            border-bottom: 1px solid var(--border);
        }
        
        tr:hover {
            background: #F0F9FF;
        }
        
        .thumbnail {
            width: 80px;
            height: 60px;
            object-fit: cover;
            border-radius: 8px;
            border: 2px solid var(--border);
            cursor: pointer;
        }
        
        .thumbnail:hover {
            transform: scale(1.05);
            border-color: var(--primary);
        }
        
        .file-icon {
            font-size: 1.8rem;
            margin-right: 10px;
        }
        
        .folder-icon { color: var(--warning); }
        .pdf-icon { color: var(--danger); }
        .image-icon { color: var(--success); }
        .file-icon-default { color: var(--primary); }
        
        .link-result {
            background: #ECFDF5;
            padding: 1.5rem;
            border-radius: 12px;
            margin: 1.5rem 0;
            border: 2px solid var(--success);
        }
        
        .file-info {
            background: #EFF6FF;
            padding: 1.5rem;
            border-radius: 12px;
            margin: 1rem 0;
            border: 2px solid var(--primary);
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
            margin-top: 1rem;
        }
        
        .info-item {
            display: flex;
            flex-direction: column;
            gap: 0.3rem;
        }
        
        .info-label {
            font-weight: 600;
            color: var(--gray);
            font-size: 0.9rem;
        }
        
        .info-value {
            font-weight: 500;
            color: var(--dark);
        }
        
        .footer {
            padding: 2.5rem;
            text-align: center;
            color: var(--gray);
            background: var(--light);
        }
        
        .action-buttons {
            display: flex;
            gap: 0.3rem;
            flex-wrap: wrap;
        }
        
        @media (max-width: 1200px) {
            .action-buttons {
                flex-direction: column;
            }
            .action-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div class="logo">{{.Name}}</div>
            <div class="subtitle">Professional File Management • Temporary Links • Auto-Sorting • Thumbnails</div>
        </div>
        
        <div class="stats">
            <div class="stat-item">📊 Files: <b>{{.Files}}</b></div>
            <div class="stat-item">📁 Folders: <b>{{.Dirs}}</b></div>
            <div class="stat-item">🖼️ Images: <b>{{.Imgs}}</b></div>
            <div class="stat-item">📦 Total Size: <b>{{.TotalSize}}</b></div>
        </div>
        
        <div class="path">
            <span>📂</span>
            <span>Path: <b>/{{.Path}}</b></span>
        </div>
        
        <div class="actions">
            <div class="upload-box">
                <form method="post" enctype="multipart/form-data" id="uploadForm">
                    <div style="display:flex;gap:15px;flex-wrap:wrap;align-items:center">
                        <input type="file" name="f" multiple required 
                               style="flex:1;min-width:250px;padding:14px;border:2px solid var(--border);border-radius:12px">
                        <button type="submit">📤 Upload Files</button>
                    </div>
                </form>
            </div>
            
            <button onclick="document.getElementById('modalFolder').style.display='flex'">📁 New Folder</button>
            <a href="/zip{{.Zip}}" class="btn btn-zip">📦 Download as ZIP</a>
            <button onclick="showBulkActions()" class="btn btn-info">🔧 Bulk Actions</button>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th style="width: 40%">Name</th>
                    <th style="width: 15%">Size</th>
                    <th style="width: 20%">Date</th>
                    <th style="width: 25%">Actions</th>
                </tr>
            </thead>
            <tbody>
                {{if ne .Path "/"}}
                <tr>
                    <td colspan="4">
                        <a href=".." style="color:var(--primary);font-weight:700;text-decoration:none;display:flex;align-items:center;gap:8px">
                            <span>⬆️</span>
                            <span>Back to Parent Directory</span>
                        </a>
                    </td>
                </tr>
                {{end}}
                
                {{range .Items}}
                <tr>
                    <td style="display:flex;align-items:center;gap:15px">
                        {{if .Thumb}}
                        <img src="{{.Thumb}}" class="thumbnail" onclick="window.open('{{$.Cur}}/{{.Name}}','_blank')">
                        {{else}}
                        <div class="file-icon {{if .Dir}}folder-icon{{else if eq .Ext ".pdf"}}pdf-icon{{else if .Img}}image-icon{{else}}file-icon-default{{end}}">
                            {{if .Dir}}📁{{else if eq .Ext ".pdf"}}📄{{else if .Img}}🖼️{{else}}📄{{end}}
                        </div>
                        {{end}}
                        
                        <div style="display:flex;flex-direction:column">
                            <a href="{{if .Dir}}{{.Name}}/{{else}}{{.Name}}{{end}}" 
                               style="color:var(--dark);text-decoration:none;font-weight:500">
                                <span style="font-weight:600">{{.Name}}</span>
                            </a>
                            <small style="color:var(--gray);font-size:0.85rem">
                                {{if .Dir}}Folder{{else if eq .Ext ".pdf"}}PDF Document{{else if .Img}}Image{{else}}File{{end}}
                                {{if not .Dir}}• {{.Ext}}{{end}}
                            </small>
                        </div>
                    </td>
                    <td>
                        <span style="font-weight:500">{{.Size}}</span>
                        {{if not .Dir}}
                        <br><small style="color:var(--gray)">{{.Bytes}} bytes</small>
                        {{end}}
                    </td>
                    <td>
                        <span style="font-weight:500">{{.Date}}</span>
                        <br><small style="color:var(--gray)">{{.Time}}</small>
                    </td>
                    <td>
                        <div class="action-buttons">
                            {{if not .Dir}}
                            <button class="action-btn btn-download" onclick="downloadFile('{{$.Cur}}/{{.Name}}')" title="Download">
                                📥
                            </button>
                            <button class="action-btn btn-link" onclick="generateTempLink('{{$.Cur}}/{{.Name}}')" title="Temporary Link">
                                🔗
                            </button>
                            {{end}}
                            <button class="action-btn btn-rename" onclick="renameItem('{{$.Cur}}/{{.Name}}', '{{.Name}}', {{.Dir}})" title="Rename">
                                ✏️
                            </button>
                            <button class="action-btn btn-info" onclick="showFileInfo('{{$.Cur}}/{{.Name}}')" title="File Info">
                                ℹ️
                            </button>
                            <button class="action-btn btn-delete" onclick="deleteItem('{{$.Cur}}/{{.Name}}', '{{.Name}}', {{.Dir}})" title="Delete">
                                🗑️
                            </button>
                        </div>
                    </td>
                </tr>
                {{end}}
            </tbody>
        </table>
        
        <div class="footer">
            <div style="margin-bottom:1rem;font-size:1.1rem;font-weight:600;color:var(--primary)">
                🚀 {{.Name}} v5.0 Professional
            </div>
            <div style="color:var(--gray);font-size:0.9rem">
                © 2025 {{.Name}} |Created with ❤️ by Fedya Serafiev
            </div>
        </div>
    </div>

    <!-- Modal Windows -->
    <div id="modalFolder" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('modalFolder').style.display='none'">×</span>
            <h2>📁 Create New Folder</h2>
            <form method="post">
                <input type="hidden" name="mkdir" value="1">
                <input name="name" placeholder="Enter folder name" required 
                       style="width:100%;padding:14px;border:2px solid var(--border);border-radius:12px;margin:1rem 0">
                <button type="submit" style="width:100%;justify-content:center">✅ Create Folder</button>
            </form>
        </div>
    </div>

    <div id="modalTempLink" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('modalTempLink').style.display='none'">×</span>
            <h2>🔗 Create Temporary Link</h2>
            <div id="linkResult"></div>
            <div style="margin-top:1.5rem;display:flex;gap:12px;flex-wrap:wrap;justify-content:center">
                <button onclick="setExpiry('1h')" class="btn-success">1 hour</button>
                <button onclick="setExpiry('24h')">24 hours</button>
                <button onclick="setExpiry('7d')" class="btn-warning">7 days</button>
                <button onclick="setExpiry('once')" class="btn-info">One-time</button>
            </div>
        </div>
    </div>

    <div id="modalRename" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('modalRename').style.display='none'">×</span>
            <h2>✏️ Rename Item</h2>
            <form id="renameForm" method="post">
                <input type="hidden" name="rename" value="1">
                <input type="hidden" id="renameOldPath" name="oldpath">
                <input type="hidden" id="renameIsDir" name="isdir">
                <input id="renameNewName" name="newname" placeholder="Enter new name" required 
                       style="width:100%;padding:14px;border:2px solid var(--border);border-radius:12px;margin:1rem 0">
                <button type="submit" style="width:100%;justify-content:center">✅ Rename</button>
            </form>
        </div>
    </div>

    <div id="modalDelete" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('modalDelete').style.display='none'">×</span>
            <h2>🗑️ Delete Item</h2>
            <p>Are you sure you want to delete "<span id="deleteItemName"></span>"?</p>
            <p style="color:var(--danger);font-weight:600;margin:1rem 0">This action cannot be undone!</p>
            <form id="deleteForm" method="post">
                <input type="hidden" name="delete" value="1">
                <input type="hidden" id="deleteItemPath" name="path">
                <div style="display:flex;gap:12px;margin-top:1.5rem">
                    <button type="button" onclick="document.getElementById('modalDelete').style.display='none'" 
                            style="flex:1;background:var(--gray)">❌ Cancel</button>
                    <button type="submit" style="flex:1" class="btn-danger">🗑️ Delete</button>
                </div>
            </form>
        </div>
    </div>

    <div id="modalFileInfo" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('modalFileInfo').style.display='none'">×</span>
            <h2>ℹ️ File Information</h2>
            <div id="fileInfoContent"></div>
        </div>
    </div>

 <script>
    let currentPath = "";
    
    function downloadFile(path) {
        window.open(path, '_blank');
    }
    
    function generateTempLink(path) {
        currentPath = path;
        document.getElementById('modalTempLink').style.display = 'flex';
        document.getElementById('linkResult').innerHTML = '<p style="text-align:center;color:var(--gray)">Select expiration time...</p>';
    }
    
    function renameItem(path, oldName, isDir) {
        document.getElementById('modalRename').style.display = 'flex';
        document.getElementById('renameOldPath').value = path;
        document.getElementById('renameIsDir').value = isDir;
        document.getElementById('renameNewName').value = oldName;
        document.getElementById('renameNewName').select();
    }
    
    function deleteItem(path, name, isDir) {
        document.getElementById('modalDelete').style.display = 'flex';
        document.getElementById('deleteItemName').textContent = name;
        document.getElementById('deleteItemPath').value = path;
    }
    
    function showFileInfo(path) {
        fetch('/info?path=' + encodeURIComponent(path))
            .then(r => r.json())
            .then(data => {
                // Generate the HTML for file-specific info only if it's not a directory
                var fileSpecificInfo = '';
                if (!data.IsDir) {
                    fileSpecificInfo = ''
                        + '<div class="info-item">'
                        + '  <span class="info-label">Extension</span>'
                        + '  <span class="info-value">' + data.Extension + '</span>'
                        + '</div>'
                        + '<div class="info-item">'
                        + '  <span class="info-label">MIME Type</span>'
                        + '  <span class="info-value">' + data.MimeType + '</span>'
                        + '</div>';
                }

                var content = ''
                    + '<div class="file-info">'
                    + '  <h3 style="margin-bottom:1rem">' + data.Name + '</h3>'
                    + '  <div class="info-grid">'
                    + '    <div class="info-item">'
                    + '      <span class="info-label">Type</span>'
                    + '      <span class="info-value">' + data.Type + '</span>'
                    + '    </div>'
                    + '    <div class="info-item">'
                    + '      <span class="info-label">Size</span>'
                    + '      <span class="info-value">' + data.Size + ' (' + data.Bytes + ' bytes)</span>'
                    + '    </div>'
                    + '    <div class="info-item">'
                    + '      <span class="info-label">Modified</span>'
                    + '      <span class="info-value">' + data.Modified + '</span>'
                    + '    </div>'
                    + '    <div class="info-item">'
                    + '      <span class="info-label">Permissions</span>'
                    + '      <span class="info-value">' + data.Permissions + '</span>'
                    + '    </div>'
                    +      fileSpecificInfo
                    + '  </div>'
                    + '</div>'
                    + '<div style="margin-top:1.5rem;display:flex;gap:12px;flex-wrap:wrap">'
                    + '  <button onclick="downloadFile(\'' + data.Path.replace(/'/g, "\\'") + '\')" class="btn-success">Download</button>'
                    + '  <button onclick="generateTempLink(\'' + data.Path.replace(/'/g, "\\'") + '\')" class="btn-info">Temporary Link</button>'
                    + '  <button onclick="renameItem(\'' + data.Path.replace(/'/g, "\\'") + '\', \'' + data.Name.replace(/'/g, "\\'") + '\', ' + (data.IsDir ? 'true' : 'false') + ')" class="btn-warning">Rename</button>'
                    + '</div>';

                document.getElementById('fileInfoContent').innerHTML = content;
                document.getElementById('modalFileInfo').style.display = 'flex';
            })
            .catch(err => {
                document.getElementById('fileInfoContent').innerHTML = '<p style="color:var(--danger)">Error loading file information</p>';
                document.getElementById('modalFileInfo').style.display = 'flex';
            });
    }
    
    function setExpiry(time) {
        fetch('/t', {
            method: 'POST',
            body: 'p=' + encodeURIComponent(currentPath) + '&e=' + time,
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' }
        })
        .then(r => r.text())
        .then(html => {
            document.getElementById('linkResult').innerHTML = html;
        });
    }
    
    function showBulkActions() {
        alert('Bulk actions feature coming soon!');
    }
    
    window.onclick = function(event) {
        if (event.target.classList.contains('modal')) {
            event.target.style.display = 'none';
        }
    }
    
    // Progress indicator for uploads
    document.getElementById('uploadForm').onsubmit = function() {
        const button = this.querySelector('button[type="submit"]');
        button.innerHTML = '<span>⏳ Uploading...</span>';
        button.disabled = true;
    };
</script>

</body>
</html>`

type Item struct {
	Name, Size, Date, Time, Ext, Thumb, Bytes string
	Dir, Img bool
}

type Data struct {
	Name, Path, Cur, Zip, TotalSize string
	Items []Item
	Files, Dirs, Imgs int
}

type FileInfo struct {
	Name        string `json:"name"`
	Path        string `json:"path"`
	Size        string `json:"size"`
	Bytes       int64  `json:"bytes"`
	Type        string `json:"type"`
	Extension   string `json:"extension"`
	MimeType    string `json:"mimeType"`
	Modified    string `json:"modified"`
	Permissions string `json:"permissions"`
	IsDir       bool   `json:"isDir"`
}

func isPathSafe(p string) bool {
	cleanedPath := filepath.Clean(p)
	return strings.HasPrefix(cleanedPath, absFileDir)
}

func main() {
	// Clean expired links
	go func() {
		for range time.Tick(10 * time.Minute) {
			now := time.Now()
			for k, v := range links {
				if now.After(v.Expiry) {
					delete(links, k)
				}
			}
		}
	}()

	http.HandleFunc("/", handler)
	http.HandleFunc("/t", tempHandler)
	http.HandleFunc("/info", infoHandler)
	http.Handle("/thumb/", http.StripPrefix("/thumb/", http.HandlerFunc(thumb)))
	
	log.Printf("🚀 Professional Server '%s' starting on port %s", serverName, port)
	log.Printf("📁 Directory: %s", absFileDir)
	log.Fatal(http.ListenAndServe(":"+port, nil))
}

func handler(w http.ResponseWriter, r *http.Request) {
	if strings.HasPrefix(r.URL.Path, "/temp/") {
		serveTemp(w, r)
		return
	}
	if strings.HasPrefix(r.URL.Path, "/thumb/") {
		thumb(w, r)
		return
	}
    
	reqPath := path.Clean(r.URL.Path)
	if reqPath == "/" {
		reqPath = ""
	}
	
	full := filepath.Join(fileDir, filepath.FromSlash(reqPath))

	if !isPathSafe(full) {
		http.Error(w, "Forbidden: Path Traversal", 403)
		return
	}

	if r.Method == "POST" {
		// Handle rename
		if r.FormValue("rename") != "" {
			oldPath := r.FormValue("oldpath")
			newName := strings.TrimSpace(r.FormValue("newname"))
			
			if newName != "" && !strings.ContainsAny(newName, "/\\") {
				oldFull := filepath.Join(fileDir, filepath.FromSlash(strings.TrimPrefix(oldPath, "/")))
				newFull := filepath.Join(filepath.Dir(oldFull), newName)
				
				if isPathSafe(oldFull) && isPathSafe(newFull) {
					os.Rename(oldFull, newFull)
				}
			}
			http.Redirect(w, r, r.URL.Path, 303)
			return
		}
		
		// Handle delete
		if r.FormValue("delete") != "" {
			delPath := r.FormValue("path")
			fullPath := filepath.Join(fileDir, filepath.FromSlash(strings.TrimPrefix(delPath, "/")))
			
			if isPathSafe(fullPath) {
				os.RemoveAll(fullPath)
			}
			http.Redirect(w, r, r.URL.Path, 303)
			return
		}
		
		// Handle folder creation
		if r.FormValue("mkdir") != "" {
			name := strings.TrimSpace(r.FormValue("name"))
			if name != "" && !strings.ContainsAny(name, "/\\") {
				os.Mkdir(filepath.Join(full, name), 0755)
			}
			http.Redirect(w, r, r.URL.Path, 303)
			return
		}

		// Handle file upload
		if err := r.ParseMultipartForm(500 << 20); err != nil {
			http.Error(w, "Upload error: "+err.Error(), 500)
			return
		}
        
		for _, h := range r.MultipartForm.File["f"] {
			f, err := h.Open()
			if err != nil {
				continue
			}
			defer f.Close()
            
			ext := strings.ToLower(filepath.Ext(h.Filename))
			dest := full
            
			// Auto-sorting
			if ext == ".pdf" {
				dest = filepath.Join(fileDir, "pdf")
				os.MkdirAll(dest, 0755)
			} else if strings.Contains(".jpg.jpeg.png.gif.webp", ext) {
				dest = filepath.Join(fileDir, "images") 
				os.MkdirAll(dest, 0755)
			}
            
			out, err := os.Create(filepath.Join(dest, h.Filename))
			if err != nil {
				continue
			}
			defer out.Close()
            
			io.Copy(out, f)
		}
		http.Redirect(w, r, r.URL.Path, 303)
		return
	}

	// Show files
	entries, err := os.ReadDir(full)
	if err != nil {
		if os.IsNotExist(err) {
			http.ServeFile(w, r, full)
		} else {
			http.Error(w, "Read error", 500)
		}
		return
	}

	var items []Item
	files, dirs, imgs := 0, 0, 0
	var totalSize int64 = 0

	// ZIP functionality
	if strings.HasPrefix(reqPath, "/zip/") {
		zipPath := filepath.Join(fileDir, filepath.FromSlash(strings.TrimPrefix(reqPath, "/zip")))
		if !isPathSafe(zipPath) {
			http.Error(w, "Forbidden", 403)
			return
		}
        
		w.Header().Set("Content-Type", "application/zip")
		w.Header().Set("Content-Disposition", fmt.Sprintf("attachment; filename=\"%s.zip\"", filepath.Base(zipPath)))

		archive := zip.NewWriter(w)
		defer archive.Close()

		filepath.Walk(zipPath, func(p string, info os.FileInfo, err error) error {
			if err != nil {
				return err
			}
            
			if p == zipPath {
				return nil
			}
            
			relPath, err := filepath.Rel(zipPath, p)
			if err != nil {
				return err
			}

			if info.IsDir() {
				_, err = archive.Create(relPath + "/")
				return err
			}

			file, err := os.Open(p)
			if err != nil {
				return err
			}
			defer file.Close()

			writer, err := archive.Create(relPath)
			if err != nil {
				return err
			}

			_, err = io.Copy(writer, file)
			return err
		})
		return
	}

	for _, e := range entries {
		info, err := e.Info()
		if err != nil {
			continue
		}
        
		item := Item{
			Name: e.Name(), 
			Date: info.ModTime().Format("02 Jan 2006"),
			Time: info.ModTime().Format("15:04:05"),
			Dir:  e.IsDir(),
		}
		
		if e.IsDir() {
			item.Size = "Folder"
			item.Bytes = "-"
			dirs++
		} else {
			size := info.Size()
			totalSize += size
			item.Bytes = formatBytes(size)
			
			switch {
			case size < 1024:
				item.Size = fmt.Sprintf("%d B", size)
			case size < 1024*1024:
				item.Size = fmt.Sprintf("%.1f KB", float64(size)/1024)
			case size < 1024*1024*1024:
				item.Size = fmt.Sprintf("%.1f MB", float64(size)/(1024*1024))
			default:
				item.Size = fmt.Sprintf("%.1f GB", float64(size)/(1024*1024*1024))
			}
			
			files++
			ext := strings.ToLower(filepath.Ext(e.Name()))
			item.Ext = ext
			if strings.Contains(".jpg.jpeg.png.gif.webp", ext) {
				imgs++
				item.Img = true
				item.Thumb = "/thumb" + path.Join("/", reqPath, e.Name())
			}
		}
		items = append(items, item)
	}

	sort.Slice(items, func(i, j int) bool {
		if items[i].Dir != items[j].Dir {
			return items[i].Dir
		}
		return strings.ToLower(items[i].Name) < strings.ToLower(items[j].Name)
	})

	totalSizeStr := formatBytes(totalSize)
	if totalSize == 0 {
		totalSizeStr = "0 B"
	}

	data := Data{
		Name:  serverName,
		Path:  reqPath,
		Cur:   "/" + reqPath,
		Zip:   reqPath,
		Items: items,
		Files: files,
		Dirs:  dirs,
		Imgs:  imgs,
		TotalSize: totalSizeStr,
	}

	tmpl.Execute(w, data)
}

func infoHandler(w http.ResponseWriter, r *http.Request) {
	pathParam := r.URL.Query().Get("path")
	if pathParam == "" {
		http.Error(w, "Path parameter required", 400)
		return
	}
	
	fullPath := filepath.Join(fileDir, filepath.FromSlash(strings.TrimPrefix(pathParam, "/")))
	if !isPathSafe(fullPath) {
		http.Error(w, "Forbidden", 403)
		return
	}
	
	info, err := os.Stat(fullPath)
	if err != nil {
		http.Error(w, "File not found", 404)
		return
	}
	
	fileInfo := FileInfo{
		Name:        filepath.Base(fullPath),
		Path:        pathParam,
		Bytes:       info.Size(),
		Modified:    info.ModTime().Format("02 Jan 2006 15:04:05"),
		Permissions: info.Mode().String(),
		IsDir:       info.IsDir(),
	}
	
	if info.IsDir() {
		fileInfo.Type = "Folder"
		fileInfo.Size = "Folder"
		fileInfo.Extension = "-"
		fileInfo.MimeType = "-"
	} else {
		ext := filepath.Ext(fullPath)
		fileInfo.Type = "File"
		fileInfo.Extension = ext
		fileInfo.MimeType = getMimeType(ext)
		
		size := info.Size()
		switch {
		case size < 1024:
			fileInfo.Size = fmt.Sprintf("%d B", size)
		case size < 1024*1024:
			fileInfo.Size = fmt.Sprintf("%.1f KB", float64(size)/1024)
		case size < 1024*1024*1024:
			fileInfo.Size = fmt.Sprintf("%.1f MB", float64(size)/(1024*1024))
		default:
			fileInfo.Size = fmt.Sprintf("%.1f GB", float64(size)/(1024*1024*1024))
		}
	}
	
	w.Header().Set("Content-Type", "application/json")
	json.NewEncoder(w).Encode(fileInfo)
}

func getMimeType(ext string) string {
	switch strings.ToLower(ext) {
	case ".pdf": return "application/pdf"
	case ".jpg", ".jpeg": return "image/jpeg"
	case ".png": return "image/png"
	case ".gif": return "image/gif"
	case ".txt": return "text/plain"
	case ".html", ".htm": return "text/html"
	case ".zip": return "application/zip"
	case ".mp4": return "video/mp4"
	case ".mp3": return "audio/mpeg"
	default: return "application/octet-stream"
	}
}

func formatBytes(bytes int64) string {
	if bytes < 1024 {
		return fmt.Sprintf("%d", bytes)
	} else if bytes < 1024*1024 {
		return fmt.Sprintf("%.0f", float64(bytes)/1024)
	} else if bytes < 1024*1024*1024 {
		return fmt.Sprintf("%.0f", float64(bytes)/(1024*1024))
	} else {
		return fmt.Sprintf("%.1f", float64(bytes)/(1024*1024*1024))
	}
}

func thumb(w http.ResponseWriter, r *http.Request) {
	p := filepath.Join(fileDir, filepath.FromSlash(strings.TrimPrefix(r.URL.Path, "/thumb")))
    
	if !isPathSafe(p) {
		return
	}
    
	f, err := os.Open(p)
	if err != nil {
		return
	}
	defer f.Close()
    
	img, _, err := image.Decode(f)
	if err != nil {
		return
	}
    
	b := img.Bounds()
	nw, nh := 100, 80
	if b.Dx() > b.Dy() {
		nh = b.Dy() * 100 / b.Dx()
	} else {
		nw = b.Dx() * 80 / b.Dy()
	}
	resized := image.NewRGBA(image.Rect(0, 0, nw, nh))
	for y := 0; y < nh; y++ {
		for x := 0; x < nw; x++ {
			resized.Set(x, y, img.At(x*b.Dx()/nw, y*b.Dy()/nh))
		}
	}
    
	w.Header().Set("Content-Type", "image/jpeg")
	jpeg.Encode(w, resized, &jpeg.Options{Quality: 85})
}

func tempHandler(w http.ResponseWriter, r *http.Request) {
	if r.Method != "POST" {
		http.Error(w, "Method not allowed", 405)
		return
	}
    
	if err := r.ParseForm(); err != nil {
		http.Error(w, "Parse error", 500)
		return
	}
    
	p := r.FormValue("p")
	e := r.FormValue("e")
    
	cleanedPath := filepath.Clean(filepath.Join(fileDir, filepath.FromSlash(strings.TrimPrefix(p, "/"))))
    
	if !isPathSafe(cleanedPath) {
		http.Error(w, "Invalid path", 400)
		return
	}

	var exp time.Time
	once := false
	switch e {
	case "1h":
		exp = time.Now().Add(1 * time.Hour)
	case "24h":
		exp = time.Now().Add(24 * time.Hour)
	case "7d":
		exp = time.Now().Add(7 * 24 * time.Hour)
	case "once":
		once = true
		exp = time.Now().Add(30 * 24 * time.Hour)
	default:
		exp = time.Now().Add(24 * time.Hour)
	}
    
	tok := make([]byte, 16)
	rand.Read(tok)
	token := hex.EncodeToString(tok)
    
	links[token] = struct {
		Path string
		Expiry time.Time
		OneTime bool
		Used bool
	}{
		Path: cleanedPath,
		Expiry: exp,
		OneTime: once,
	}
    
	link := fmt.Sprintf("http://%s/temp/%s", r.Host, token)
	fmt.Fprintf(w, `<div class="link-result">Done! Valid until <b>%s</b><br><br>
		<input value="%s" readonly style="width:100%%;padding:12px;border-radius:8px;border:1px solid #10B981" onclick="this.select()">
		<br><small>Ctrl+C to copy</small></div>`, exp.Format("02 Jan 2006 15:04"), link)
}

func serveTemp(w http.ResponseWriter, r *http.Request) {
	token := strings.TrimPrefix(r.URL.Path, "/temp/")
	l, ok := links[token]
	
	if !ok || time.Now().After(l.Expiry) || (l.OneTime && l.Used) {
		http.Error(w, "Link expired", 410)
		return
	}

	if !isPathSafe(l.Path) {
		http.Error(w, "Invalid link", 403)
		return
	}
    
	if l.OneTime {
		links[token] = struct {
			Path string
			Expiry time.Time
			OneTime bool
			Used bool
		}{
			Path: l.Path,
			Expiry: l.Expiry,
			OneTime: true,
			Used: true,
		}
	}
    
	http.ServeFile(w, r, l.Path)
}
EOF

# Създаване на Go модул и компилиране
echo 'module fedya' > go.mod
echo -e "\e[1;33m🔨 Compiling Professional Server...\e[0m"

if ! CGO_ENABLED=0 go build -ldflags="-s -w" -o fedya-pro 2>&1; then
    echo -e "\e[1;31m❌ Compilation error!\e[0m"
    exit 1
fi

# Инсталиране на изпълнимия файл
install -m 755 fedya-pro /usr/local/bin/fedya-pro

# Спиране на старата услуга
systemctl stop fedya-pro 2>/dev/null || true

if $SERVICE; then
    # Създаване на systemd услуга
    cat > /etc/systemd/system/fedya-pro.service <<EOF
[Unit]
Description=$NAME - Professional File Server
After=network.target

[Service]
Type=simple
Environment=FILE_DIR=$DIR
Environment=PORT=$PORT
Environment=SERVER_NAME=$NAME
ExecStart=/usr/local/bin/fedya-pro
WorkingDirectory=$DIR
Restart=always
RestartSec=3
User=root

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable fedya-pro
    systemctl start fedya-pro
    
    echo -e "\e[1;32m✅ Service is active!\e[0m"
    sleep 2
    echo -e "\e[1;33m📊 Service status:\e[0m"
    systemctl status fedya-pro --no-pager -l
else
    echo -e "\n\e[1;33m📝 Manual start:\e[0m"
    echo -e "\e[1;36mFILE_DIR='$DIR' PORT='$PORT' SERVER_NAME='$NAME' /usr/local/bin/fedya-pro\e[0m"
fi

# Финална информация
IP=$(hostname -I | awk '{print $1}')
echo
echo -e "\e[1;32m🎉 FEDYA PRO FILE SERVER v5.0 IS READY!\e[0m"
echo
echo -e "\e[1;36m🌐 Access the server:\e[0m"
echo -e "   \e[1;33mhttp://$IP:$PORT\e[0m"
echo
echo -e "\e[1;35m🚀 PROFESSIONAL FEATURES:\e[0m"
echo -e "   📥 Download files with one click"
echo -e "   ✏️ Rename files and folders" 
echo -e "   🗑️ Delete items with confirmation"
echo -e "   ℹ️ Detailed file information"
echo -e "   🔗 Temporary links with expiry"
echo -e "   📊 File size in bytes and human-readable"
echo -e "   📁 Folder creation"
echo -e "   📦 ZIP download of folders"
echo -e "   🖼️ Image thumbnails"
echo -e "   📤 File upload with progress"
echo -e "   🎨 Professional color-coded buttons"
echo -e "   📱 Responsive design for all devices"
echo -e "   🔧 Bulk actions (coming soon)"
echo
echo -e "\e[1;33m💡 Tip: Hover over buttons to see tooltips with descriptions!\e[0m"

# Почистване
rm -rf "$TMP"