#!/bin/bash

# Цветове за терминала
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Функция за проверка на root потребител
check_root() {
    if [ "$(id -u)" -ne 0 ]; then
        echo -e "${RED}Грешка: Този скрипт трябва да се изпълнява с root права!${NC}" >&2
        exit 1
    fi
}

# Функция за проверка на интернет връзка
check_internet() {
    echo -e "${BLUE}Проверка на интернет връзка...${NC}"
    if ! ping -c 1 google.com &> /dev/null; then
        echo -e "${RED}Грешка: Няма интернет връзка! Моля, проверете мрежата.${NC}"
        exit 1
    fi
    echo -e "${GREEN}Интернет връзката е налична.${NC}"
}

# Функция за инсталиране на Docker
install_docker() {
    echo -e "${BLUE}Проверка за инсталиран Docker...${NC}"
    if ! command -v docker &> /dev/null; then
        echo -e "${YELLOW}Docker не е инсталиран. Започва инсталация...${NC}"
        
        # Инсталиране на необходими пакети
        apt-get update
        apt-get install -y apt-transport-https ca-certificates curl software-properties-common

        # Добавяне на Docker GPG ключ
        curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker-archive-keyring.gpg

        # Добавяне на Docker repository
        echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker-archive-keyring.gpg] https://download.docker.com/linux/ubuntu $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null

        # Инсталиране на Docker
        apt-get update
        apt-get install -y docker-ce docker-ce-cli containerd.io

        # Проверка на Docker инсталацията
        if docker --version &> /dev/null; then
            echo -e "${GREEN}Docker е успешно инсталиран.${NC}"
        else
            echo -e "${RED}Грешка при инсталация на Docker!${NC}"
            exit 1
        fi
    else
        echo -e "${GREEN}Docker вече е инсталиран.${NC}"
    fi
}

# Функция за инсталиране на Docker Compose
install_docker_compose() {
    echo -e "${BLUE}Проверка за инсталиран Docker Compose...${NC}"
    if ! command -v docker-compose &> /dev/null; then
        echo -e "${YELLOW}Docker Compose не е инсталиран. Започва инсталация...${NC}"
        
        # Изтегляне на Docker Compose
        curl -L "https://github.com/docker/compose/releases/download/1.29.2/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
        
        # Права за изпълнение
        chmod +x /usr/local/bin/docker-compose
        
        # Проверка на инсталацията
        if docker-compose --version &> /dev/null; then
            echo -e "${GREEN}Docker Compose е успешно инсталиран.${NC}"
        else
            echo -e "${RED}Грешка при инсталация на Docker Compose!${NC}"
            exit 1
        fi
    else
        echo -e "${GREEN}Docker Compose вече е инсталиран.${NC}"
    fi
}

# Функция за създаване на Docker Compose файл за Nextcloud
create_docker_compose() {
    echo -e "${BLUE}Създаване на Docker Compose файл за Nextcloud...${NC}"
    
    cat > docker-compose.yml <<EOF
version: '3'

services:
  db:
    image: mariadb:10.5
    restart: always
    command: --transaction-isolation=READ-COMMITTED --binlog-format=ROW
    volumes:
      - db_data:/var/lib/mysql
    environment:
      - MYSQL_ROOT_PASSWORD=nextcloud_root_password
      - MYSQL_PASSWORD=nextcloud_password
      - MYSQL_DATABASE=nextcloud
      - MYSQL_USER=nextcloud

  app:
    image: nextcloud
    restart: always
    ports:
      - 8080:80
    links:
      - db
    volumes:
      - nextcloud_data:/var/www/html
    environment:
      - MYSQL_PASSWORD=nextcloud_password
      - MYSQL_DATABASE=nextcloud
      - MYSQL_USER=nextcloud
      - MYSQL_HOST=db

volumes:
  db_data:
  nextcloud_data:
EOF

    echo -e "${GREEN}Docker Compose файлът е създаден успешно.${NC}"
}

# Функция за стартиране на Nextcloud
start_nextcloud() {
    echo -e "${BLUE}Стартиране на Nextcloud с Docker Compose...${NC}"
    docker-compose up -d
    
    echo -e "${GREEN}Nextcloud е стартиран успешно!${NC}"
    echo -e "${YELLOW}Можете да достъпите Nextcloud на: http://$(hostname -I | awk '{print $1}'):8080${NC}"
    echo -e "${YELLOW}Потребителско име: admin${NC}"
    echo -e "${YELLOW}Парола: ще бъде зададена при първо влизане${NC}"
}

# Главна функция
main() {
    echo -e "${BLUE}╔════════════════════════════════════════╗${NC}"
    echo -e "${BLUE}║   ИНСТАЛАЦИЯ НА NEXTCLOUD С DOCKER     ║${NC}"
    echo -e "${BLUE}╚════════════════════════════════════════╝${NC}"
    
    check_root
    check_internet
    install_docker
    install_docker_compose
    create_docker_compose
    start_nextcloud
    
    echo -e "${GREEN}Инсталацията завърши успешно!${NC}"
}

# Стартиране на главната функция
main