#!/bin/bash

# Конфигурационен файл за втори Proxmox
CONFIG_FILE="$(dirname "$0")/backup_config_pve2.conf"
LOG_FILE="$(dirname "$0")/backup_sync_pve2.log"
DOWNLOADED_FILE="$(dirname "$0")/.downloaded_backups_pve2"

# Функция за логване
log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# Проверка дали конфигурационният файл съществува
if [[ ! -f "$CONFIG_FILE" ]]; then
    log "ERROR: Конфигурационният файл $CONFIG_FILE не съществува!"
    cat > "$CONFIG_FILE" << 'EOF'
# Конфигурация за Proxmox backup sync - PVE2
PROXMOX_HOST=10.110.110.110
PROXMOX_USER=root
REMOTE_BACKUP_PATH=/mnt/backup/dump
LOCAL_BACKUP_PATH=/mnt/d/backup_proxmox_pve2

# LXC контейнери за синхронизиране (разделени със запетая)
# Пример: LXC_IDS=101,102,103
LXC_IDS=

# SSH опции
SSH_OPTIONS="-o ConnectTimeout=30 -o ServerAliveInterval=60"

# Максимален брой дни за задържане на локални бекъпи (0 = безкрайно)
KEEP_DAYS=30
EOF
    log "Създаден е примерен конфигурационен файл: $CONFIG_FILE"
    log "Моля, редактирайте го според вашите нужди и стартирайте отново скрипта."
    exit 1
fi

# Зареждане на конфигурацията
source "$CONFIG_FILE"

# Създаване на локална директория ако не съществува
mkdir -p "$LOCAL_BACKUP_PATH"

# Създаване на файл за проследяване на свалените бекъпи
touch "$DOWNLOADED_FILE"

log "Започвам проверка за нови бекъпи от $PROXMOX_HOST..."
log "Наблюдавани LXC контейнери: $LXC_IDS"

# Функция за получаване на списък с отдалечени бекъпи
get_remote_backups() {
    local lxc_id=$1
    ssh $SSH_OPTIONS "${PROXMOX_USER}@${PROXMOX_HOST}" \
        "find ${REMOTE_BACKUP_PATH} -name 'vzdump-lxc-${lxc_id}-*.tar.zst' -type f -printf '%f\n'" 2>/dev/null
}

# Функция за проверка дали бекъп е вече свален
is_downloaded() {
    local backup_name=$1
    grep -Fxq "$backup_name" "$DOWNLOADED_FILE"
}

# Функция за маркиране на бекъп като свален
mark_as_downloaded() {
    local backup_name=$1
    echo "$backup_name" >> "$DOWNLOADED_FILE"
}

# Функция за изтриване на стари локални бекъпи
cleanup_old_backups() {
    if [[ "$KEEP_DAYS" -gt 0 ]]; then
        log "Изтривам локални бекъпи по-стари от $KEEP_DAYS дни..."
        find "$LOCAL_BACKUP_PATH" -name "vzdump-lxc-*.tar.zst" -type f -mtime +$KEEP_DAYS -delete
    fi
}

# Главен цикъл за всеки LXC контейнер
IFS=',' read -ra LXC_ARRAY <<< "$LXC_IDS"
for lxc_id in "${LXC_ARRAY[@]}"; do
    lxc_id=$(echo "$lxc_id" | xargs) # Премахване на интервали
    log "Проверявам за нови бекъпи на LXC $lxc_id..."

    # Получаване на списък с отдалечени бекъпи
    remote_backups=$(get_remote_backups "$lxc_id")

    if [[ -z "$remote_backups" ]]; then
        log "Няма намерени бекъпи за LXC $lxc_id"
        continue
    fi

    # Проверка за всеки бекъп
    while IFS= read -r backup_file; do
        [[ -z "$backup_file" ]] && continue

        if ! is_downloaded "$backup_file"; then
            log "Намерен нов бекъп: $backup_file"
            log "Започвам сваляне..."

            # Изпълнение на rsync
            if rsync -avz --partial --progress -e "ssh $SSH_OPTIONS" \
                "${PROXMOX_USER}@${PROXMOX_HOST}:${REMOTE_BACKUP_PATH}/${backup_file}" \
                "${LOCAL_BACKUP_PATH}/"; then

                log "Успешно свален: $backup_file"
                mark_as_downloaded "$backup_file"
            else
                log "ERROR: Грешка при сваляне на $backup_file"
            fi
        else
            log "Бекъп $backup_file вече е свален, прескачам..."
        fi
    done <<< "$remote_backups"
done

# Почистване на стари бекъпи
cleanup_old_backups

log "Завършена проверка за нови бекъпи от $PROXMOX_HOST."