#!/bin/bash
#
# Автор: Федя Серафиев / urocibg.eu

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

log()   { echo -e "${GREEN}[ИНФО]${NC} $1"; }
warn()  { echo -e "${YELLOW}[ПРЕДУПРЕЖДЕНИЕ]${NC} $1"; }
error() { echo -e "${RED}[ГРЕШКА]${NC} $1"; exit 1; }

[ "$EUID" -ne 0 ] && error "Стартирай със sudo!"

log "Fedya's File Server – Професионална версия с красив дизайн"

read -p "Директория за файловете [/var/www/files]: " file_dir
file_dir=${file_dir:-"/var/www/files"}
mkdir -p "$file_dir"

read -p "Порт [8080]: " port
port=${port:-"8080"}

read -p "Име на сървъра [Fedya File Server]: " server_name
server_name=${server_name:-"Fedya File Server"}

read -p "Стартиране като услуга? [Y/n]: " svc
[[ "$svc" =~ ^[Nn]$ ]] && auto_start=false || auto_start=true

log "Инсталиране на Go..."
command -v go >/dev/null || { apt update && apt install -y golang-go; }

tmp=$(mktemp -d)
cd "$tmp"

go mod init fedya-files >/dev/null 2>&1

cat > main.go <<'EOF'
package main

import (
	"archive/zip"
	"fmt"
	"html/template"
	"image"
	"image/jpeg"
	_ "image/png"
	_ "image/gif"
	"io"
	"log"
	"net/http"
	"os"
	"path/filepath"
	"sort"
	"strings"
	"time"
)

var (
	fileDir    = getEnv("FILE_DIR", "/var/www/files")
	port       = getEnv("PORT", "8080")
	serverName = getEnv("SERVER_NAME", "Fedya File Server")
	tmpl       = template.Must(template.New("index").Parse(htmlTemplate))
)

func getEnv(key, defaultValue string) string {
	if value := os.Getenv(key); value != "" {
		return value
	}
	return defaultValue
}

const htmlTemplate = `<!DOCTYPE html>
<html lang="bg">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{.ServerName}}</title>
    <link rel="icon" href="data:image/svg+xml,<svg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 100 100'><defs><linearGradient id='grad' x1='0%' y1='0%' x2='100%' y2='100%'><stop offset='0%' style='stop-color:%233b82f6;stop-opacity:1' /><stop offset='100%' style='stop-color:%231e40af;stop-opacity:1' /></linearGradient></defs><circle cx='50' cy='50' r='45' fill='url(%23grad)'/><rect x='30' y='30' width='40' height='30' rx='5' fill='white'/><rect x='35' y='45' width='30' height='2' fill='%233b82f6'/><rect x='35' y='50' width='25' height='2' fill='%233b82f6'/><rect x='35' y='55' width='20' height='2' fill='%233b82f6'/></svg>">
    <style>
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --primary-light: #60a5fa;
            --secondary: #8b5cf6;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --dark: #1e293b;
            --light: #f8fafc;
            --gray: #64748b;
            --border: #e2e8f0;
            --shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1);
        }
        
        * { 
            margin: 0; 
            padding: 0; 
            box-sizing: border-box; 
        }
        
        body {
            font-family: 'Inter', 'Segoe UI', system-ui, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: var(--dark);
            min-height: 100vh;
            padding: 20px;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1400px;
            margin: 0 auto;
            background: white;
            border-radius: 24px;
            box-shadow: var(--shadow), 0 20px 80px rgba(0,0,0,0.15);
            overflow: hidden;
            backdrop-filter: blur(10px);
        }
        
        header {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 3rem 2rem;
            text-align: center;
            position: relative;
            overflow: hidden;
        }
        
        header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url("data:image/svg+xml,%3Csvg width='100' height='100' viewBox='0 0 100 100' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M11 18c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm48 25c3.866 0 7-3.134 7-7s-3.134-7-7-7-7 3.134-7 7 3.134 7 7 7zm-43-7c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm63 31c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zM34 90c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zm56-76c1.657 0 3-1.343 3-3s-1.343-3-3-3-3 1.343-3 3 1.343 3 3 3zM12 86c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm28-65c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm23-11c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zm-6 60c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm29 22c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zM32 63c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zm57-13c2.76 0 5-2.24 5-5s-2.24-5-5-5-5 2.24-5 5 2.24 5 5 5zm-9-21c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2zM60 91c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2zM35 41c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2zM12 60c1.105 0 2-.895 2-2s-.895-2-2-2-2 .895-2 2 .895 2 2 2z' fill='%23ffffff' fill-opacity='0.1' fill-rule='evenodd'/%3E%3C/svg%3E");
            animation: float 20s infinite linear;
        }
        
        @keyframes float {
            0% { transform: translateX(0) translateY(0); }
            100% { transform: translateX(-100px) translateY(-100px); }
        }
        
        .logo-container {
            position: relative;
            z-index: 2;
        }
        
        .logo {
            font-size: 5rem;
            margin-bottom: 1rem;
            filter: drop-shadow(0 8px 16px rgba(0,0,0,0.3));
            background: linear-gradient(135deg, #ffffff, #e0f2fe);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        h1 {
            margin: 0;
            font-size: 3rem;
            font-weight: 800;
            letter-spacing: -1px;
            text-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        
        .subtitle {
            opacity: 0.95;
            margin-top: 0.75rem;
            font-weight: 400;
            font-size: 1.3rem;
            max-width: 600px;
            margin-left: auto;
            margin-right: auto;
        }
        
        .path-bar {
            background: linear-gradient(90deg, #f1f5f9, #f8fafc);
            padding: 1.25rem 2rem;
            font-family: 'JetBrains Mono', 'Monaco', 'Consolas', monospace;
            font-size: 1.1rem;
            color: var(--primary-dark);
            border-bottom: 1px solid var(--border);
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .path-bar::before {
            content: '📍';
            font-size: 1.2rem;
        }
        
        .actions {
            padding: 2rem;
            background: #f8fafc;
            display: flex;
            gap: 1.5rem;
            flex-wrap: wrap;
            align-items: center;
            justify-content: space-between;
            border-bottom: 1px solid var(--border);
        }
        
        .upload-box {
            background: white;
            padding: 1.75rem;
            border-radius: 20px;
            box-shadow: var(--shadow);
            flex: 1;
            min-width: 350px;
            border: 2px dashed var(--primary-light);
            transition: all 0.3s ease;
        }
        
        .upload-box:hover {
            border-color: var(--primary);
            transform: translateY(-2px);
            box-shadow: 0 15px 40px rgba(59, 130, 246, 0.15);
        }
        
        .upload-form {
            display: flex;
            gap: 15px;
            width: 100%;
            align-items: center;
        }
        
        .file-input {
            flex: 1;
            padding: 14px 18px;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            background: white;
            font-size: 1rem;
            transition: all 0.3s ease;
            font-family: inherit;
        }
        
        .file-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .file-input:hover {
            border-color: var(--primary-light);
        }
        
        button, .btn {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
            border: none;
            padding: 14px 32px;
            border-radius: 14px;
            cursor: pointer;
            font-weight: 600;
            font-size: 1.05rem;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 10px;
            transition: all 0.3s ease;
            font-family: inherit;
            position: relative;
            overflow: hidden;
        }
        
        button::before, .btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left 0.5s;
        }
        
        button:hover::before, .btn:hover::before {
            left: 100%;
        }
        
        button:hover, .btn:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 25px rgba(59, 130, 246, 0.4);
        }
        
        button:active, .btn:active {
            transform: translateY(-1px);
        }
        
        .btn-success {
            background: linear-gradient(135deg, var(--success), #059669);
        }
        
        .btn-success:hover {
            box-shadow: 0 10px 25px rgba(16, 185, 129, 0.4);
        }
        
        .file-table {
            width: 100%;
            border-collapse: collapse;
            margin: 0;
        }
        
        th {
            background: linear-gradient(135deg, #f8fafc, #f1f5f9);
            padding: 1.5rem 2rem;
            text-align: left;
            color: var(--gray);
            font-weight: 700;
            border-bottom: 3px solid var(--border);
            font-size: 1.1rem;
        }
        
        td {
            padding: 1.5rem 2rem;
            border-bottom: 1px solid var(--border);
            transition: all 0.3s ease;
            vertical-align: top;
        }
        
        tr {
            transition: all 0.3s ease;
        }
        
        tr:hover {
            background: linear-gradient(90deg, #f0f9ff, #f8fafc);
            transform: translateX(8px);
        }
        
        .size-col {
            text-align: right;
            color: var(--gray);
            font-variant-numeric: tabular-nums;
            font-weight: 600;
        }
        
        .date-col {
            color: #94a3b8;
            font-size: 0.95rem;
            font-weight: 500;
        }
        
        .folder {
            font-weight: 700;
            color: var(--primary);
            font-size: 1.1rem;
        }
        
        .file-link {
            color: var(--dark);
            text-decoration: none;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 12px;
            font-weight: 500;
        }
        
        .file-link:hover {
            color: var(--primary);
            transform: translateX(5px);
        }
        
        .icon {
            font-size: 1.4rem;
            transition: transform 0.3s ease;
        }
        
        .file-link:hover .icon {
            transform: scale(1.2);
        }
        
        .back-link {
            color: var(--primary);
            text-decoration: none;
            font-weight: 700;
            display: flex;
            align-items: center;
            gap: 10px;
            padding: 12px 0;
            font-size: 1.1rem;
            transition: all 0.3s ease;
        }
        
        .back-link:hover {
            transform: translateX(-5px);
        }
        
        .thumbnail {
            width: 90px;
            height: 70px;
            object-fit: cover;
            border-radius: 10px;
            border: 3px solid #e2e8f0;
            margin-right: 15px;
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }
        
        .thumbnail:hover {
            transform: scale(1.05);
            border-color: var(--primary);
            box-shadow: 0 8px 20px rgba(0,0,0,0.15);
        }
        
        .file-info {
            display: flex;
            align-items: center;
        }
        
        .file-name {
            flex: 1;
        }
        
        .stats {
            display: flex;
            gap: 2rem;
            padding: 1.5rem 2rem;
            background: #f8fafc;
            border-bottom: 1px solid var(--border);
            font-size: 0.95rem;
            color: var(--gray);
        }
        
        .stat-item {
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        footer {
            text-align: center;
            padding: 2.5rem;
            color: #94a3b8;
            font-size: 1rem;
            border-top: 1px solid var(--border);
            background: #f8fafc;
        }
        
        .footer-content {
            display: flex;
            justify-content: space-between;
            align-items: center;
            max-width: 1000px;
            margin: 0 auto;
        }
        
        @media (max-width: 768px) {
            .actions {
                flex-direction: column;
                align-items: stretch;
            }
            
            .upload-form {
                flex-direction: column;
            }
            
            th, td {
                padding: 1.25rem;
            }
            
            .thumbnail {
                width: 70px;
                height: 55px;
            }
            
            h1 {
                font-size: 2.2rem;
            }
            
            .footer-content {
                flex-direction: column;
                gap: 1rem;
            }
        }
        
        .loading {
            opacity: 0.7;
            pointer-events: none;
        }
        
        .pulse {
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { opacity: 1; }
            50% { opacity: 0.7; }
            100% { opacity: 1; }
        }
    </style>
</head>
<body>
<div class="container">
    <header>
        <div class="logo-container">
            <div class="logo">📁</div>
            <h1>{{.ServerName}}</h1>
            <div class="subtitle">Професионално споделяне на файлове с миниатюри</div>
        </div>
    </header>
    
    <div class="stats">
        <div class="stat-item">📊 Общо файлове: <strong>{{.FileCount}}</strong></div>
        <div class="stat-item">📁 Папки: <strong>{{.DirCount}}</strong></div>
        <div class="stat-item">🖼️ Картинки: <strong>{{.ImageCount}}</strong></div>
    </div>
    
    <div class="path-bar">
        Път: <strong>/{{.Current}}</strong>
    </div>
    
    <div class="actions">
        <div class="upload-box">
            <form method="post" enctype="multipart/form-data" class="upload-form" id="uploadForm">
                <input type="file" name="f" required class="file-input" id="fileInput">
                <button type="submit" id="uploadBtn">📤 Качи файл</button>
            </form>
        </div>
        <a href="/zip{{.CurrentPath}}" class="btn btn-success">📦 Изтегли като ZIP</a>
    </div>
    
    <table class="file-table">
        <thead>
            <tr>
                <th>Име</th>
                <th class="size-col">Размер</th>
                <th class="date-col">Дата</th>
            </tr>
        </thead>
        <tbody>
            {{if ne .Current "/"}}
            <tr>
                <td colspan="3">
                    <a href=".." class="back-link">⬆️ Назад</a>
                </td>
            </tr>
            {{end}}
            {{range .Files}}
            <tr>
                <td>
                    {{if .IsDir}}
                    <div class="file-info">
                        <span class="folder">📁 {{.Name}}</span>
                        <a href="{{.Name}}/" class="file-link">[Отвори]</a>
                    </div>
                    {{else}}
                    <div class="file-info">
                        {{if .Thumbnail}}
                        <img src="{{.Thumbnail}}" alt="{{.Name}}" class="thumbnail" 
                             onerror="this.style.display='none'" 
                             onclick="window.open('{{$.CurrentPath}}/{{.Name}}', '_blank')"
                             style="cursor: zoom-in;">
                        {{end}}
                        <div class="file-name">
                            <a href="{{.Name}}" class="file-link">
                                <span class="icon">{{if .Thumbnail}}🖼️{{else}}📄{{end}}</span>
                                {{.Name}}
                            </a>
                        </div>
                    </div>
                    {{end}}
                </td>
                <td class="size-col">{{.Size}}</td>
                <td class="date-col">{{.ModTime}}</td>
            </tr>
            {{end}}
        </tbody>
    </table>
    
    <footer>
        <div class="footer-content">
            <div>© 2024 {{.ServerName}} - Професионален файлов сървър</div>
            <div>🔄 {{.CurrentTime}}</div>
        </div>
    </footer>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const thumbnails = document.querySelectorAll('.thumbnail');
    thumbnails.forEach(thumb => {
        thumb.addEventListener('error', function() {
            this.style.display = 'none';
        });
    });
    
    const fileInput = document.querySelector('#fileInput');
    const uploadForm = document.querySelector('#uploadForm');
    const uploadBtn = document.querySelector('#uploadBtn');
    
    fileInput.addEventListener('change', function() {
        if (this.files[0] && this.files[0].size > 100 * 1024 * 1024) {
            if (!confirm('Файлът е голям (' + (this.files[0].size / (1024*1024)).toFixed(1) + 'MB). Сигурни ли сте, че искате да го качите?')) {
                this.value = '';
            }
        }
    });
    
    uploadForm.addEventListener('submit', function(e) {
        uploadBtn.innerHTML = '⏳ Качване...';
        uploadBtn.classList.add('loading', 'pulse');
        uploadBtn.disabled = true;
    });
    
    // Add some interactive effects
    const rows = document.querySelectorAll('tr');
    rows.forEach(row => {
        row.addEventListener('mouseenter', function() {
            this.style.transition = 'all 0.3s ease';
        });
    });
});
</script>
</body>
</html>`

type File struct {
	Name      string
	Size      string
	ModTime   string
	IsDir     bool
	Thumbnail string
}

type PageData struct {
	Files       []File
	Current     string
	CurrentPath string
	ServerName  string
	FileCount   int
	DirCount    int
	ImageCount  int
	CurrentTime string
}

func main() {
	log.Printf("Стартиране на %s на порт %s за директория %s", serverName, port, fileDir)
	
	http.HandleFunc("/", handler)
	http.HandleFunc("/zip", zipHandler)
	http.HandleFunc("/thumb/", thumbHandler)
	
	log.Printf("Сървърът е готов на http://0.0.0.0:%s", port)
	log.Fatal(http.ListenAndServe(":"+port, nil))
}

func handler(w http.ResponseWriter, r *http.Request) {
	if strings.HasPrefix(r.URL.Path, "/zip") {
		zipHandler(w, r)
		return
	}

	if strings.HasPrefix(r.URL.Path, "/thumb/") {
		thumbHandler(w, r)
		return
	}

	reqPath := filepath.Clean(r.URL.Path)
	if reqPath == "/" || reqPath == "." {
		reqPath = ""
	}
	fullPath := filepath.Join(fileDir, reqPath)

	if !strings.HasPrefix(fullPath, fileDir) {
		http.Error(w, "Достъпът е отказан", http.StatusForbidden)
		return
	}

	fi, err := os.Stat(fullPath)
	if err != nil {
		http.NotFound(w, r)
		return
	}

	if fi.IsDir() {
		if r.Method == "POST" {
			upload(w, r, fullPath)
			return
		}
		listDir(w, reqPath, fullPath)
		return
	}

	http.ServeFile(w, r, fullPath)
}

func upload(w http.ResponseWriter, r *http.Request, dir string) {
	if err := r.ParseMultipartForm(500 << 20); err != nil {
		http.Error(w, "Файлът е твърде голям (максимум 500MB)", http.StatusBadRequest)
		return
	}

	file, header, err := r.FormFile("f")
	if err != nil {
		http.Error(w, "Грешка при качване", http.StatusBadRequest)
		return
	}
	defer file.Close()

	filePath := filepath.Join(dir, header.Filename)
	dst, err := os.Create(filePath)
	if err != nil {
		http.Error(w, "Грешка при създаване на файл", http.StatusInternalServerError)
		return
	}
	defer dst.Close()

	if _, err := io.Copy(dst, file); err != nil {
		http.Error(w, "Грешка при запис на файл", http.StatusInternalServerError)
		return
	}

	http.Redirect(w, r, r.URL.Path, http.StatusSeeOther)
}

func listDir(w http.ResponseWriter, relPath, fullPath string) {
	entries, err := os.ReadDir(fullPath)
	if err != nil {
		http.Error(w, "Грешка при четене на директория", http.StatusInternalServerError)
		return
	}

	var files []File
	fileCount := 0
	dirCount := 0
	imageCount := 0

	for _, e := range entries {
		info, err := e.Info()
		if err != nil {
			continue
		}

		size := "📁 папка"
		if !e.IsDir() {
			fileCount++
			s := info.Size()
			switch {
			case s < 1024:
				size = fmt.Sprintf("%d Б", s)
			case s < 1048576:
				size = fmt.Sprintf("%.1f КБ", float64(s)/1024)
			case s < 1073741824:
				size = fmt.Sprintf("%.1f МБ", float64(s)/1048576)
			default:
				size = fmt.Sprintf("%.1f ГБ", float64(s)/1073741824)
			}
		} else {
			dirCount++
		}

		thumbnail := ""
		if !e.IsDir() && isImageFile(e.Name()) {
			imageCount++
			thumbnail = fmt.Sprintf("/thumb%s/%s", relPath, e.Name())
		}

		files = append(files, File{
			Name:      e.Name(),
			Size:      size,
			ModTime:   info.ModTime().Format("02.01.2006 15:04"),
			IsDir:     e.IsDir(),
			Thumbnail: thumbnail,
		})
	}

	sort.Slice(files, func(i, j int) bool {
		if files[i].IsDir != files[j].IsDir {
			return files[i].IsDir
		}
		return strings.ToLower(files[i].Name) < strings.ToLower(files[j].Name)
	})

	current := relPath
	if current == "" {
		current = "/"
	}

	currentPath := relPath
	if currentPath != "" && !strings.HasPrefix(currentPath, "/") {
		currentPath = "/" + currentPath
	}

	data := PageData{
		Files:       files,
		Current:     current,
		CurrentPath: currentPath,
		ServerName:  serverName,
		FileCount:   fileCount,
		DirCount:    dirCount,
		ImageCount:  imageCount,
		CurrentTime: time.Now().Format("02.01.2006 15:04:05"),
	}

	if err := tmpl.Execute(w, data); err != nil {
		http.Error(w, "Грешка при генериране на страница", http.StatusInternalServerError)
	}
}

func thumbHandler(w http.ResponseWriter, r *http.Request) {
	path := strings.TrimPrefix(r.URL.Path, "/thumb")
	fullPath := filepath.Join(fileDir, filepath.Clean(path))

	if !strings.HasPrefix(fullPath, fileDir) {
		http.Error(w, "Достъпът е отказан", http.StatusForbidden)
		return
	}

	if _, err := os.Stat(fullPath); os.IsNotExist(err) || !isImageFile(fullPath) {
		http.NotFound(w, r)
		return
	}

	generateThumbnail(w, fullPath)
}

func generateThumbnail(w http.ResponseWriter, filePath string) {
	file, err := os.Open(filePath)
	if err != nil {
		http.Error(w, "Грешка при отваряне на файл", http.StatusInternalServerError)
		return
	}
	defer file.Close()

	img, _, err := image.Decode(file)
	if err != nil {
		http.Error(w, "Невалидно изображение", http.StatusInternalServerError)
		return
	}

	bounds := img.Bounds()
	width := bounds.Dx()
	height := bounds.Dy()

	newWidth := 90
	newHeight := 70

	if width > height {
		newHeight = int(float64(height) * float64(newWidth) / float64(width))
	} else {
		newWidth = int(float64(width) * float64(newHeight) / float64(height))
	}

	dst := image.NewRGBA(image.Rect(0, 0, newWidth, newHeight))
	
	for y := 0; y < newHeight; y++ {
		for x := 0; x < newWidth; x++ {
			srcX := x * width / newWidth
			srcY := y * height / newHeight
			dst.Set(x, y, img.At(srcX, srcY))
		}
	}

	w.Header().Set("Content-Type", "image/jpeg")
	if err := jpeg.Encode(w, dst, &jpeg.Options{Quality: 85}); err != nil {
		http.Error(w, "Грешка при кодиране на миниатюра", http.StatusInternalServerError)
		return
	}
}

func isImageFile(filename string) bool {
	ext := strings.ToLower(filepath.Ext(filename))
	imageExts := []string{".jpg", ".jpeg", ".png", ".gif", ".bmp", ".webp", ".tiff", ".svg"}
	
	for _, imgExt := range imageExts {
		if ext == imgExt {
			return true
		}
	}
	return false
}

func zipHandler(w http.ResponseWriter, r *http.Request) {
	path := strings.TrimPrefix(r.URL.Path, "/zip")
	if path == "" {
		path = "/"
	}
	
	fullPath := filepath.Join(fileDir, filepath.Clean(path))

	if !strings.HasPrefix(fullPath, fileDir) {
		http.Error(w, "Достъпът е отказан", http.StatusForbidden)
		return
	}

	if _, err := os.Stat(fullPath); os.IsNotExist(err) {
		http.NotFound(w, r)
		return
	}

	w.Header().Set("Content-Type", "application/zip")
	filename := filepath.Base(fullPath)
	if filename == "" || filename == "." || filename == "/" {
		filename = "files"
	}
	w.Header().Set("Content-Disposition", fmt.Sprintf("attachment; filename=\"%s.zip\"", filename))

	zw := zip.NewWriter(w)
	defer zw.Close()

	filepath.Walk(fullPath, func(filePath string, info os.FileInfo, err error) error {
		if err != nil {
			return err
		}

		relPath, err := filepath.Rel(fullPath, filePath)
		if err != nil {
			return err
		}

		if relPath == "." {
			return nil
		}

		header, err := zip.FileInfoHeader(info)
		if err != nil {
			return err
		}

		header.Name = relPath

		if info.IsDir() {
			header.Name += "/"
		} else {
			header.Method = zip.Deflate
		}

		writer, err := zw.CreateHeader(header)
		if err != nil {
			return err
		}

		if !info.IsDir() {
			file, err := os.Open(filePath)
			if err != nil {
				return err
			}
			defer file.Close()

			_, err = io.Copy(writer, file)
			if err != nil {
				return err
			}
		}

		return nil
	})
}
EOF

log "Компилиране на професионалната версия..."
CGO_ENABLED=0 go build -ldflags="-s -w" -o fedya-server || error "Грешка при компилация!"

install -m 755 fedya-server /usr/local/bin/fedya-server

# Спиране на старите услуги
systemctl stop file-server sabork-server fedya-server 2>/dev/null || true
systemctl disable file-server sabork-server 2>/dev/null || true

if $auto_start; then
    log "Конфигуриране на systemd услуга..."
    
    cat > /etc/systemd/system/fedya-server.service <<EOF
[Unit]
Description=$server_name - Professional File Server
After=network.target
StartLimitIntervalSec=0

[Service]
Type=simple
Environment=FILE_DIR=$file_dir
Environment=PORT=$port
Environment=SERVER_NAME="$server_name"
ExecStart=/usr/local/bin/fedya-server
WorkingDirectory=$file_dir
Restart=always
RestartSec=3

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable fedya-server >/dev/null 2>&1
    
    if systemctl start fedya-server; then
        log "Услугата $server_name е стартирана успешно!"
        sleep 2
        log "Статус на услугата:"
        systemctl status fedya-server --no-pager -l
    else
        error "Неуспешно стартиране на услугата. Провери: journalctl -u fedya-server -f"
    fi
else
    log "Ръчно стартиране:"
    echo "FILE_DIR='$file_dir' PORT='$port' SERVER_NAME='$server_name' /usr/local/bin/fedya-server"
fi

rm -rf "$tmp"
IP=$(hostname -I | awk '{print $1}')
log "ГОТОВО! Отвори: http://$IP:$port"
log "🎉 Професионален дизайн • Цветно лого • Фавикон • Статистики"

if $auto_start; then
    echo
    log "Команди за управление:"
    echo "  systemctl status fedya-server"
    echo "  journalctl -u fedya-server -f"
    echo "  systemctl restart fedya-server"
fi

echo
log "💼 Сървърът е готов за колегите!"
echo "   🌐 Може да го споделиш на: http://$IP:$port"
echo "   📊 Има статистики за файлове, папки и картинки"
echo "   🎨 Професионален дизайн с анимации и ефекти"
echo "   🔒 Безопасен и стабилен за корпоративна среда"