#!/bin/bash

# Ubuntu Server Initial Setup Script
# Автор: Fedya Serafiev
# Дата: 16 май 2025
# Версия: 2.1
# 
# Този скрипт извършва основните задачи за първоначална настройка на Ubuntu Server
# Включва: актуализация на системата, настройка на защитна стена, инсталиране на полезни пакети,
# конфигуриране на SSH, създаване на нов потребител с административни права и базова защита

# Цветове за терминала
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
MAGENTA='\033[0;35m'
NC='\033[0m' # No Color

# Проверка за root привилегии
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}Този скрипт трябва да се изпълни като root потребител${NC}"
   echo -e "${YELLOW}Моля, стартирайте го отново с: sudo $0${NC}"
   exit 1
fi

# Функция за показване на прогрес
show_progress() {
    echo -e "${BLUE}================================================${NC}"
    echo -e "${GREEN}>>> $1${NC}"
    echo -e "${BLUE}================================================${NC}"
}

# Функция за проверка на успех
check_success() {
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}[УСПЕХ] $1${NC}"
    else
        echo -e "${RED}[ГРЕШКА] $2${NC}"
        if [ "$3" = "exit" ]; then
            exit 1
        fi
    fi
}

# Функция за пауза
pause() {
    echo -e "${YELLOW}Натиснете Enter за да продължите...${NC}"
    read -r
}

# Генериране на системна информация
generate_system_info() {
    local hostname=$(hostname)
    local ip_address=$(hostname -I | awk '{print $1}')
    local date=$(date +"%Y-%m-%d %H:%M:%S")
    local os_info=$(lsb_release -d | cut -f2-)
    local kernel=$(uname -r)
    local uptime=$(uptime -p)
    
    echo -e "${CYAN}*************************************************************************${NC}"
    echo -e "${MAGENTA}                          ВНИМАНИЕ!${NC}"
    echo -e "${YELLOW}     Този сървър е частна собственост. Неоторизираният достъп"
    echo -e "     е забранен и подлежи на наказателна отговорност."
    echo -e "     Всички действия се записват и следят.${NC}"
    echo -e ""
    echo -e "${GREEN}     Системна информация:${NC}"
    echo -e "     * Хост: $hostname"
    echo -e "     * IP адрес: $ip_address"
    echo -e "     * Дата: $date"
    echo -e "     * ОС: $os_info"
    echo -e "     * Ядро: $kernel"
    echo -e "     * Uptime: $uptime"
    echo -e "${CYAN}*************************************************************************${NC}"
}

# Задаване на времева зона (по подразбиране е Europe/Sofia)
set_timezone() {
    show_progress "Настройка на времева зона на Europe/Sofia"
    timedatectl set-timezone Europe/Sofia
    check_success "Времевата зона е настроена" "Грешка при настройка на времева зона"
    echo -e "${YELLOW}Текуща времева зона: $(timedatectl | grep "Time zone" | awk '{print $3}')${NC}"
}

# Актуализиране на системата
update_system() {
    show_progress "Актуализиране на системните пакети"
    echo -e "${YELLOW}Извършване на apt update...${NC}"
    apt update -q
    check_success "Репотарите са актуализирани" "Грешка при актуализация на репотарите"
    
    echo -e "${YELLOW}Извършване на apt upgrade...${NC}"
    DEBIAN_FRONTEND=noninteractive apt upgrade -y -q
    check_success "Пакетите са обновени" "Грешка при обновяване на пакети"
    
    echo -e "${YELLOW}Извършване на apt dist-upgrade...${NC}"
    DEBIAN_FRONTEND=noninteractive apt dist-upgrade -y -q
    check_success "Дистрибутивните обновления са приложени" "Грешка при dist-upgrade"
    
    echo -e "${YELLOW}Извършване на apt autoremove...${NC}"
    apt autoremove -y -q
    apt autoclean -q
    check_success "Излишните пакети са премахнати" "Грешка при премахване на излишни пакети"
}

# Инсталиране на основни пакети
install_essential_packages() {
    show_progress "Инсталиране на полезни пакети"
    
    # Основен списък с пакети
    BASE_PACKAGES=(
        curl wget vim git htop net-tools nmap tmux unzip zip
        ufw fail2ban logwatch ncdu tree dnsutils bash-completion
        apt-transport-https ca-certificates gnupg-agent software-properties-common
        jq rsync screen iftop iotop lsof strace sysstat
    )
    
    # Инсталиране на основни пакети
    echo -e "${YELLOW}Инсталиране на основни пакети...${NC}"
    apt install -y -q "${BASE_PACKAGES[@]}"
    check_success "Основните пакети са инсталирани" "Грешка при инсталиране на основни пакети"
    
    # Допълнителни пакети за Docker (ако е нужно)
    read -p "Искате ли да инсталирате Docker? (y/n): " -n 1 -r INSTALL_DOCKER
    echo
    if [[ $INSTALL_DOCKER =~ ^[Yy]$ ]]; then
        echo -e "${YELLOW}Инсталиране на Docker...${NC}"
        curl -fsSL https://download.docker.com/linux/ubuntu/gpg | apt-key add -
        add-apt-repository "deb [arch=amd64] https://download.docker.com/linux/ubuntu $(lsb_release -cs) stable"
        apt update -q
        apt install -y -q docker-ce docker-ce-cli containerd.io
        check_success "Docker е инсталиран" "Грешка при инсталиране на Docker"
        
        # Добавяне на текущия потребител към групата docker
        if [ -n "$SUDO_USER" ]; then
            usermod -aG docker $SUDO_USER
            check_success "Потребителят $SUDO_USER е добавен към групата docker" "Грешка при добавяне към docker група"
        fi
    fi
    
    # Допълнителни пакети за мониторинг
    read -p "Искате ли да инсталирате пакети за мониторинг (cockpit, netdata)? (y/n): " -n 1 -r INSTALL_MONITORING
    echo
    if [[ $INSTALL_MONITORING =~ ^[Yy]$ ]]; then
        echo -e "${YELLOW}Инсталиране на cockpit...${NC}"
        apt install -y -q cockpit
        systemctl enable --now cockpit.socket
        check_success "Cockpit е инсталиран и активиран" "Грешка при инсталиране на cockpit"
        
        echo -e "${YELLOW}Инсталиране на netdata...${NC}"
        bash <(curl -Ss https://my-netdata.io/kickstart.sh) --non-interactive
        check_success "Netdata е инсталиран" "Грешка при инсталиране на netdata"
    fi
}

# Конфигуриране на защитна стена (UFW)
configure_firewall() {
    show_progress "Конфигуриране на защитната стена (UFW)"
    
    # Разрешаване на SSH
    ufw allow ssh
    check_success "SSH портът е разрешен" "Грешка при разрешаване на SSH порт"
    
    # Питане за разрешаване на други портове
    read -p "Искате ли да разрешите HTTP трафик (порт 80)? (y/n): " -n 1 -r HTTP_RESPONSE
    echo
    if [[ $HTTP_RESPONSE =~ ^[Yy]$ ]]; then
        ufw allow 80/tcp
        check_success "HTTP трафик разрешен" "Грешка при разрешаване на HTTP порт"
    fi
    
    read -p "Искате ли да разрешите HTTPS трафик (порт 443)? (y/n): " -n 1 -r HTTPS_RESPONSE
    echo
    if [[ $HTTPS_RESPONSE =~ ^[Yy]$ ]]; then
        ufw allow 443/tcp
        check_success "HTTPS трафик разрешен" "Грешка при разрешаване на HTTPS порт"
    fi
    
    read -p "Искате ли да разрешите други портове? (y/n): " -n 1 -r OTHER_PORTS
    echo
    if [[ $OTHER_PORTS =~ ^[Yy]$ ]]; then
        read -p "Въведете портове, разделени с интервал (напр. 8080 8443): " -a PORTS_ARRAY
        for port in "${PORTS_ARRAY[@]}"; do
            ufw allow "$port/tcp"
            check_success "Порт $port/tcp е разрешен" "Грешка при разрешаване на порт $port"
        done
    fi
    
    # Активиране на защитната стена
    ufw --force enable
    check_success "Защитната стена е активирана" "Грешка при активиране на защитната стена"
    
    echo -e "${YELLOW}Текущ статус на защитната стена:${NC}"
    ufw status verbose
}

# Конфигуриране на Fail2ban
configure_fail2ban() {
    show_progress "Конфигуриране на Fail2ban за защита от brute-force атаки"
    
    # Проверка дали fail2ban е инсталиран
    if ! command -v fail2ban-server &> /dev/null; then
        apt install -y -q fail2ban
        check_success "Fail2ban е инсталиран" "Грешка при инсталиране на fail2ban"
    fi
    
    # Създаване на конфигурационен файл, ако не съществува
    if [ ! -f /etc/fail2ban/jail.local ]; then
        cp /etc/fail2ban/jail.conf /etc/fail2ban/jail.local
    fi
    
    # Добавяне на базова конфигурация
    cat << EOF >> /etc/fail2ban/jail.local

# Потребителска конфигурация
[sshd]
enabled = true
port = ssh
filter = sshd
logpath = /var/log/auth.log
maxretry = 5
bantime = 3600
findtime = 600
ignoreip = 127.0.0.1/8 ::1

[sshd-ddos]
enabled = true
port = ssh
filter = sshd
logpath = /var/log/auth.log
maxretry = 5
findtime = 600
bantime = 86400
EOF
    
    # Рестартиране на услугата
    systemctl restart fail2ban
    check_success "Fail2ban услугата е рестартирана" "Грешка при рестартиране на fail2ban"
    
    systemctl enable fail2ban
    check_success "Fail2ban е конфигуриран да се стартира автоматично" "Грешка при активиране на fail2ban"
    
    echo -e "${YELLOW}Текущ статус на Fail2ban:${NC}"
    fail2ban-client status
}

# Създаване на нов администраторски потребител
create_admin_user() {
    show_progress "Създаване на нов потребител с sudo права"
    
    read -p "Въведете име за новия потребител: " NEW_USER
    
    # Проверка дали потребителят вече съществува
    if id "$NEW_USER" &>/dev/null; then
        echo -e "${YELLOW}Потребителят $NEW_USER вече съществува!${NC}"
        read -p "Искате ли да добавите потребителя към групата sudo? (y/n): " -n 1 -r ADD_SUDO
        echo
        if [[ $ADD_SUDO =~ ^[Yy]$ ]]; then
            usermod -aG sudo $NEW_USER
            check_success "Потребителят $NEW_USER е добавен към групата sudo" "Грешка при добавяне към sudo група"
        fi
    else
        # Създаване на потребителя
        adduser --gecos "" $NEW_USER
        check_success "Потребителят $NEW_USER е създаден" "Грешка при създаване на потребител"
        
        # Добавяне към групата sudo
        usermod -aG sudo $NEW_USER
        check_success "Потребителят $NEW_USER е добавен към групата sudo" "Грешка при добавяне към sudo група"
        
        # Настройка на SSH ключове за новия потребител
        read -p "Искате ли да настроите SSH ключове за $NEW_USER? (y/n): " -n 1 -r SSH_KEY_RESPONSE
        echo
        
        if [[ $SSH_KEY_RESPONSE =~ ^[Yy]$ ]]; then
            mkdir -p /home/$NEW_USER/.ssh
            chmod 700 /home/$NEW_USER/.ssh
            
            read -p "Въведете публичния SSH ключ: " SSH_KEY
            echo $SSH_KEY > /home/$NEW_USER/.ssh/authorized_keys
            chmod 600 /home/$NEW_USER/.ssh/authorized_keys
            chown -R $NEW_USER:$NEW_USER /home/$NEW_USER/.ssh
            check_success "SSH ключът е добавен за потребителя $NEW_USER" "Грешка при добавяне на SSH ключ"
        fi
    fi
    
    # Забрана за вход с парола за root и новия потребител
    if [ -f /etc/ssh/sshd_config ]; then
        sed -i "/^#*PermitRootLogin/c\PermitRootLogin prohibit-password" /etc/ssh/sshd_config
        
        if grep -q "^#*PasswordAuthentication" /etc/ssh/sshd_config; then
            sed -i "/^#*PasswordAuthentication/c\PasswordAuthentication no" /etc/ssh/sshd_config
        else
            echo "PasswordAuthentication no" >> /etc/ssh/sshd_config
        fi
        
        systemctl restart sshd
        check_success "SSH конфигурацията е обновена" "Грешка при обновяване на SSH конфигурация"
    fi
}

# Подобряване на конфигурацията на SSH
harden_ssh() {
    show_progress "Подобряване на SSH конфигурацията"
    
    # Създаване на резервно копие на SSH конфигурационния файл
    cp /etc/ssh/sshd_config /etc/ssh/sshd_config.bak
    check_success "Резервно копие на sshd_config е създадено" "Грешка при създаване на резервно копие"
    
    # Настройки за по-добра сигурност
    sed -i 's/#PermitRootLogin yes/PermitRootLogin prohibit-password/' /etc/ssh/sshd_config
    sed -i 's/#PasswordAuthentication yes/PasswordAuthentication no/' /etc/ssh/sshd_config
    sed -i 's/X11Forwarding yes/X11Forwarding no/' /etc/ssh/sshd_config
    sed -i 's/#MaxAuthTries 6/MaxAuthTries 3/' /etc/ssh/sshd_config
    sed -i 's/#ClientAliveInterval 0/ClientAliveInterval 300/' /etc/ssh/sshd_config
    sed -i 's/#ClientAliveCountMax 3/ClientAliveCountMax 0/' /etc/ssh/sshd_config
    
    # Добавяне на допълнителни настройки за сигурност
    if ! grep -q "^UsePAM" /etc/ssh/sshd_config; then
        echo "UsePAM yes" >> /etc/ssh/sshd_config
    fi
    
    if ! grep -q "^PrintLastLog" /etc/ssh/sshd_config; then
        echo "PrintLastLog yes" >> /etc/ssh/sshd_config
    fi
    
    if ! grep -q "^AllowAgentForwarding" /etc/ssh/sshd_config; then
        echo "AllowAgentForwarding no" >> /etc/ssh/sshd_config
    fi
    
    if ! grep -q "^AllowTcpForwarding" /etc/ssh/sshd_config; then
        echo "AllowTcpForwarding no" >> /etc/ssh/sshd_config
    fi
    
    read -p "Желаете ли да промените SSH порта? (y/n): " -n 1 -r CHANGE_PORT
    echo
    
    if [[ $CHANGE_PORT =~ ^[Yy]$ ]]; then
        read -p "Въведете нов SSH порт (препоръчително > 1024): " NEW_SSH_PORT
        
        # Проверка за валиден порт
        if [[ "$NEW_SSH_PORT" =~ ^[0-9]+$ ]] && [ "$NEW_SSH_PORT" -ge 1 ] && [ "$NEW_SSH_PORT" -le 65535 ]; then
            sed -i "s/#Port 22/Port $NEW_SSH_PORT/" /etc/ssh/sshd_config
            
            # Разрешаване на новия порт в защитната стена
            ufw allow $NEW_SSH_PORT/tcp
            ufw delete allow ssh
            
            check_success "SSH портът е променен на $NEW_SSH_PORT" "Грешка при промяна на SSH порт"
            echo -e "${YELLOW}ВАЖНО: Не затваряйте текущата SSH сесия, докато не проверите, че можете да се свържете на новия порт!${NC}"
        else
            echo -e "${RED}Невалиден порт. Пропускане на промяната.${NC}"
        fi
    fi
    
    # Рестартиране на SSH услугата
    systemctl restart sshd
    check_success "SSH услугата е рестартирана" "Грешка при рестартиране на SSH"
}

# Настройка на автоматични актуализации
setup_automatic_updates() {
    show_progress "Настройка на автоматични обновления"
    
    apt install -y -q unattended-upgrades apt-listchanges
    
    # Конфигуриране на автоматичните обновления
    cat << EOF > /etc/apt/apt.conf.d/50unattended-upgrades
Unattended-Upgrade::Origins-Pattern {
    "origin=Debian,codename=\${distro_codename}-updates";
    "origin=Debian,codename=\${distro_codename}-proposed-updates";
    "origin=Debian,codename=\${distro_codename},label=Debian";
    "origin=Debian,codename=\${distro_codename},label=Debian-Security";
    "origin=Ubuntu,codename=\${distro_codename}-updates";
    "origin=Ubuntu,codename=\${distro_codename}-proposed-updates";
    "origin=Ubuntu,codename=\${distro_codename},label=Ubuntu";
    "origin=Ubuntu,codename=\${distro_codename},label=UbuntuESM";
    "origin=Ubuntu,codename=\${distro_codename}-security";
};
Unattended-Upgrade::Package-Blacklist {
};
Unattended-Upgrade::DevRelease "false";
Unattended-Upgrade::Remove-Unused-Dependencies "true";
Unattended-Upgrade::Automatic-Reboot "true";
Unattended-Upgrade::Automatic-Reboot-Time "02:00";
Unattended-Upgrade::AutoFixInterruptedDpkg "true";
EOF

    # Активиране на автоматичните обновления
    cat << EOF > /etc/apt/apt.conf.d/20auto-upgrades
APT::Periodic::Update-Package-Lists "1";
APT::Periodic::Download-Upgradeable-Packages "1";
APT::Periodic::AutocleanInterval "7";
APT::Periodic::Unattended-Upgrade "1";
EOF

    systemctl restart unattended-upgrades
    check_success "Автоматичните обновления са конфигурирани" "Грешка при конфигуриране на автоматични обновления"
    
    echo -e "${YELLOW}Текущ статус на автоматичните обновления:${NC}"
    systemctl status unattended-upgrades.service --no-pager
}

# Настройка на swap файл (ако е необходимо)
configure_swap() {
    show_progress "Проверка за swap пространство"
    
    # Проверка на наличния swap
    CURRENT_SWAP=$(free -m | awk '/^Swap:/ {print $2}')
    TOTAL_MEM=$(free -m | awk '/^Mem:/ {print $2}')
    RECOMMENDED_SWAP=$((TOTAL_MEM * 2))
    
    if [ "$CURRENT_SWAP" -eq 0 ]; then
        read -p "Няма конфигуриран swap. Искате ли да създадете swap файл? (y/n): " -n 1 -r CREATE_SWAP
        echo
        
        if [[ $CREATE_SWAP =~ ^[Yy]$ ]]; then
            read -p "Въведете размер на swap файла в GB (рекомендация: $RECOMMENDED_SWAP MB за $TOTAL_MEM MB RAM): " SWAP_SIZE
            
            # Създаване на swap файл
            fallocate -l ${SWAP_SIZE}G /swapfile
            chmod 600 /swapfile
            mkswap /swapfile
            swapon /swapfile
            
            # Добавяне на swap файла в /etc/fstab
            echo '/swapfile none swap sw 0 0' >> /etc/fstab
            
            # Настройка на swappiness
            echo 'vm.swappiness=10' >> /etc/sysctl.conf
            echo 'vm.vfs_cache_pressure=50' >> /etc/sysctl.conf
            sysctl -p
            
            check_success "Swap файл от ${SWAP_SIZE}GB е създаден и конфигуриран" "Грешка при създаване на swap файл"
        fi
    else
        echo -e "${YELLOW}Системата вече има ${CURRENT_SWAP}MB swap пространство${NC}"
        echo -e "${YELLOW}Обща RAM памет: ${TOTAL_MEM}MB${NC}"
        echo -e "${YELLOW}Рекомендуван swap размер: ${RECOMMENDED_SWAP}MB${NC}"
        
        read -p "Искате ли да промените swap настройките? (y/n): " -n 1 -r CHANGE_SWAP
        echo
        
        if [[ $CHANGE_SWAP =~ ^[Yy]$ ]]; then
            read -p "Въведете нова стойност за vm.swappiness (по подразбиране 10): " SWAPPINESS
            read -p "Въведете нова стойност за vm.vfs_cache_pressure (по подразбиране 50): " CACHE_PRESSURE
            
            sed -i '/vm.swappiness/d' /etc/sysctl.conf
            sed -i '/vm.vfs_cache_pressure/d' /etc/sysctl.conf
            
            echo "vm.swappiness=$SWAPPINESS" >> /etc/sysctl.conf
            echo "vm.vfs_cache_pressure=$CACHE_PRESSURE" >> /etc/sysctl.conf
            sysctl -p
            
            check_success "Swap настройките са променени" "Грешка при промяна на swap настройки"
        fi
    fi
}

# Настройка на NTP (Network Time Protocol)
configure_ntp() {
    show_progress "Настройка на NTP за синхронизация на времето"
    
    # Инсталиране на NTP
    apt install -y -q chrony
    
    # Конфигуриране на Chrony
    cat << EOF > /etc/chrony/chrony.conf
pool ntp.ubuntu.com iburst
keyfile /etc/chrony/chrony.keys
driftfile /var/lib/chrony/chrony.drift
logdir /var/log/chrony
maxupdateskew 100.0
hwclockfile /etc/adjtime
rtcsync
makestep 1 3
EOF
    
    # Рестартиране на услугата
    systemctl restart chrony
    check_success "Chrony услугата е рестартирана" "Грешка при рестартиране на chrony"
    
    systemctl enable chrony
    check_success "Chrony е конфигуриран да се стартира автоматично" "Грешка при активиране на chrony"
    
    echo -e "${YELLOW}Текущ статус на NTP синхронизацията:${NC}"
    chronyc tracking
    chronyc sources
}

# Добавяне на основен банер
add_banner() {
    show_progress "Добавяне на системен банер"
    
    # Създаване на банер файл с динамична системна информация
    cat << 'EOF' > /etc/issue.net
*************************************************************************
                          ВНИМАНИЕ!
     Този сървър е частна собственост. Неоторизираният достъп
     е забранен и подлежи на наказателна отговорност.
     Всички действия се записват и следят.
     
     Системна информация:
EOF

    # Добавяне на динамична информация
    echo "     * Хост: $(hostname)" >> /etc/issue.net
    echo "     * IP адрес: $(hostname -I | awk '{print $1}')" >> /etc/issue.net
    echo "     * Дата: $(date +"%Y-%m-%d %H:%M:%S")" >> /etc/issue.net
    echo "     * ОС: $(lsb_release -d | cut -f2-)" >> /etc/issue.net
    echo "     * Ядро: $(uname -r)" >> /etc/issue.net
    echo "     * Uptime: $(uptime -p)" >> /etc/issue.net
    echo "*************************************************************************" >> /etc/issue.net

    # Активиране на банера в SSH
    if grep -q "^#Banner" /etc/ssh/sshd_config; then
        sed -i 's|^#Banner.*|Banner /etc/issue.net|' /etc/ssh/sshd_config
    else
        echo "Banner /etc/issue.net" >> /etc/ssh/sshd_config
    fi
    
    systemctl restart sshd
    check_success "Системният банер е конфигуриран" "Грешка при конфигуриране на банер"
    
    # Показване на банера
    echo -e "${CYAN}Системният банер ще изглежда така:${NC}"
    generate_system_info
}

# Настройка на базови sysctl параметри
configure_sysctl() {
    show_progress "Настройка на базови системни параметри"
    
    # Създаване на резервно копие
    cp /etc/sysctl.conf /etc/sysctl.conf.bak
    
    # Добавяне на параметри за подобрена сигурност и производителност
    cat << EOF >> /etc/sysctl.conf

# Настройки за мрежова сигурност
net.ipv4.conf.all.rp_filter=1
net.ipv4.conf.default.rp_filter=1
net.ipv4.conf.all.accept_redirects=0
net.ipv4.conf.default.accept_redirects=0
net.ipv4.conf.all.secure_redirects=0
net.ipv4.conf.default.secure_redirects=0
net.ipv4.icmp_echo_ignore_broadcasts=1
net.ipv4.icmp_ignore_bogus_error_responses=1
net.ipv4.tcp_syncookies=1
net.ipv4.tcp_max_syn_backlog=2048
net.ipv4.tcp_synack_retries=2
net.ipv4.tcp_syn_retries=2

# Настройки за мрежова производителност
net.ipv4.tcp_window_scaling=1
net.ipv4.tcp_keepalive_time=600
net.ipv4.tcp_keepalive_probes=5
net.ipv4.tcp_keepalive_intvl=15
net.ipv4.tcp_fin_timeout=30
net.ipv4.tcp_tw_reuse=1
net.ipv4.tcp_max_tw_buckets=1440000

# Настройки за защита от DDoS атаки
net.ipv4.tcp_syncookies=1
net.ipv4.netfilter.ip_conntrack_max=655360
EOF
    
    # Прилагане на промените
    sysctl -p
    check_success "Sysctl параметрите са конфигурирани" "Грешка при конфигуриране на sysctl параметри"
}

# Инсталиране и конфигуриране на ClamAV (антивирус)
install_clamav() {
    show_progress "Инсталиране на ClamAV антивирус"
    
    apt install -y -q clamav clamav-daemon
    
    # Обновяване на вирусните дефиниции
    freshclam
    
    # Конфигуриране на автоматично сканиране
    cat << EOF > /etc/cron.daily/clamav-scan
#!/bin/bash
SCAN_DIR="/home /var/www"
LOG_FILE="/var/log/clamav/scan.log"
/usr/bin/clamscan -i -r $SCAN_DIR >> $LOG_FILE
EOF
    
    chmod +x /etc/cron.daily/clamav-scan
    
    systemctl restart clamav-daemon
    systemctl enable clamav-daemon
    
    check_success "ClamAV е инсталиран и конфигуриран" "Грешка при инсталиране на ClamAV"
}

# Настройка на journald за по-добро логване
configure_journald() {
    show_progress "Настройка на системното логване"
    
    # Конфигуриране на journald
    cat << EOF > /etc/systemd/journald.conf
[Journal]
Storage=persistent
Compress=yes
SystemMaxUse=1G
SystemMaxFileSize=100M
SystemMaxFiles=10
EOF
    
    systemctl restart systemd-journald
    check_success "Системното логване е конфигурирано" "Грешка при конфигуриране на journald"
}

# Инсталиране и конфигуриране на RKHunter (rootkit detector)
install_rkhunter() {
    show_progress "Инсталиране на RKHunter за откриване на rootkits"
    
    apt install -y -q rkhunter
    
    # Конфигуриране на RKHunter
    sed -i 's/UPDATE_MIRRORS=0/UPDATE_MIRRORS=1/' /etc/rkhunter.conf
    sed -i 's/MIRRORS_MODE=1/MIRRORS_MODE=0/' /etc/rkhunter.conf
    sed -i 's/CRON_DAILY_RUN=""/CRON_DAILY_RUN="true"/' /etc/default/rkhunter
    sed -i 's/CRON_DB_UPDATE=""/CRON_DB_UPDATE="true"/' /etc/default/rkhunter
    
    # Обновяване на базата данни
    rkhunter --update
    rkhunter --propupd
    
    # Изпълнение на проверка
    rkhunter --check --sk
    
    check_success "RKHunter е инсталиран и конфигуриран" "Грешка при инсталиране на RKHunter"
}

# Настройка на мрежови параметри
configure_network() {
    show_progress "Настройка на мрежови параметри"
    
    # Конфигуриране на hosts файла
    cat << EOF > /etc/hosts
127.0.0.1       localhost
127.0.1.1       $(hostname)

# IPv6
::1             localhost ip6-localhost ip6-loopback
ff02::1         ip6-allnodes
ff02::2         ip6-allrouters
EOF
    
    # Конфигуриране на hostname
    read -p "Искате ли да промените hostname на сървъра? (текущо: $(hostname)) (y/n): " -n 1 -r CHANGE_HOSTNAME
    echo
    
    if [[ $CHANGE_HOSTNAME =~ ^[Yy]$ ]]; then
        read -p "Въведете ново hostname: " NEW_HOSTNAME
        hostnamectl set-hostname $NEW_HOSTNAME
        check_success "Hostname е променен на $NEW_HOSTNAME" "Грешка при промяна на hostname"
    fi
    
    # Конфигуриране на SSMTP за изпращане на имейли
    if command -v apt &> /dev/null; then
        apt install -y -q ssmtp mailutils
    fi
    
    check_success "Мрежовите параметри са конфигурирани" "Грешка при конфигуриране на мрежови параметри"
}

# Създаване на базов backup скрипт
create_backup_script() {
    show_progress "Създаване на базов backup скрипт"
    
    cat << 'EOF' > /usr/local/bin/backup-server.sh
#!/bin/bash

# Базов backup скрипт за сървър
# Автор: Fedya Serafiev
# Дата: $(date +%Y-%m-%d)

BACKUP_DIR="/backups"
LOG_FILE="/var/log/backup.log"
DATE=$(date +%Y-%m-%d_%H-%M-%S)
BACKUP_NAME="backup_$DATE.tar.gz"

# Създаване на директория за backup, ако не съществува
mkdir -p $BACKUP_DIR

# Функция за логване
log() {
    echo "[$(date +'%Y-%m-%d %H:%M:%S')] $1" >> $LOG_FILE
}

log "Стартиране на backup процеса"

# Създаване на backup
tar -czf $BACKUP_DIR/$BACKUP_NAME \
    /etc \
    /home \
    /var/www \
    /var/log \
    /var/lib/mysql 2>/dev/null

if [ $? -eq 0 ]; then
    log "Backup успешно създаден: $BACKUP_NAME"
else
    log "Грешка при създаване на backup"
    exit 1
fi

# Изтриване на стари backups (по-стари от 7 дни)
find $BACKUP_DIR -type f -name "backup_*.tar.gz" -mtime +7 -exec rm -f {} \;
log "Изтрити са стари backups (по-стари от 7 дни)"

log "Backup процесът завърши успешно"
EOF
    
    chmod +x /usr/local/bin/backup-server.sh
    
    # Добавяне към cron за автоматично изпълнение
    if [ ! -f /etc/cron.d/backup-job ]; then
        echo "0 2 * * * root /usr/local/bin/backup-server.sh" > /etc/cron.d/backup-job
    fi
    
    check_success "Backup скриптът е създаден" "Грешка при създаване на backup скрипт"
}

# Основно меню с опции
main_menu() {
    while true; do
        clear
        generate_system_info
        echo -e "${BLUE}==============================================${NC}"
        echo -e "${GREEN}      UBUNTU SERVER НАЧАЛНА НАСТРОЙКА         ${NC}"
        echo -e "${BLUE}==============================================${NC}"
        echo -e "Моля, изберете операциите, които искате да изпълните:"
        echo ""
        echo -e "${YELLOW}1) Настройка на времева зона (Europe/Sofia)${NC}"
        echo -e "${YELLOW}2) Актуализиране на системата${NC}"
        echo -e "${YELLOW}3) Инсталиране на основни пакети${NC}"
        echo -e "${YELLOW}4) Конфигуриране на защитна стена (UFW)${NC}"
        echo -e "${YELLOW}5) Конфигуриране на Fail2ban${NC}"
        echo -e "${YELLOW}6) Създаване на нов администраторски потребител${NC}"
        echo -e "${YELLOW}7) Подобряване на SSH конфигурацията${NC}"
        echo -e "${YELLOW}8) Настройка на автоматични актуализации${NC}"
        echo -e "${YELLOW}9) Настройка на swap пространство${NC}"
        echo -e "${YELLOW}10) Настройка на NTP (синхронизация на времето)${NC}"
        echo -e "${YELLOW}11) Добавяне на системен банер${NC}"
        echo -e "${YELLOW}12) Настройка на системни параметри (sysctl)${NC}"
        echo -e "${YELLOW}13) Инсталиране на ClamAV антивирус${NC}"
        echo -e "${YELLOW}14) Настройка на системно логване${NC}"
        echo -e "${YELLOW}15) Инсталиране на RKHunter${NC}"
        echo -e "${YELLOW}16) Настройка на мрежови параметри${NC}"
        echo -e "${YELLOW}17) Създаване на backup скрипт${NC}"
        echo ""
        echo -e "${GREEN}A) Изпълни всички основни операции (1-11)${NC}"
        echo -e "${GREEN}B) Изпълни всички разширени операции (12-17)${NC}"
        echo -e "${GREEN}C) Изпълни всички операции (1-17)${NC}"
        echo -e "${RED}Q) Изход${NC}"
        echo -e "${BLUE}==============================================${NC}"
        
        read -p "Въведете вашия избор: " USER_CHOICE
        
        case $USER_CHOICE in
            1) set_timezone; pause ;;
            2) update_system; pause ;;
            3) install_essential_packages; pause ;;
            4) configure_firewall; pause ;;
            5) configure_fail2ban; pause ;;
            6) create_admin_user; pause ;;
            7) harden_ssh; pause ;;
            8) setup_automatic_updates; pause ;;
            9) configure_swap; pause ;;
            10) configure_ntp; pause ;;
            11) add_banner; pause ;;
            12) configure_sysctl; pause ;;
            13) install_clamav; pause ;;
            14) configure_journald; pause ;;
            15) install_rkhunter; pause ;;
            16) configure_network; pause ;;
            17) create_backup_script; pause ;;
            [Aa]) 
                set_timezone
                update_system
                install_essential_packages
                configure_firewall
                configure_fail2ban
                create_admin_user
                harden_ssh
                setup_automatic_updates
                configure_swap
                configure_ntp
                add_banner
                echo -e "${GREEN}Всички основни операции са изпълнени успешно!${NC}"
                pause
                ;;
            [Bb])
                configure_sysctl
                install_clamav
                configure_journald
                install_rkhunter
                configure_network
                create_backup_script
                echo -e "${GREEN}Всички разширени операции са изпълнени успешно!${NC}"
                pause
                ;;
            [Cc]) 
                set_timezone
                update_system
                install_essential_packages
                configure_firewall
                configure_fail2ban
                create_admin_user
                harden_ssh
                setup_automatic_updates
                configure_swap
                configure_ntp
                add_banner
                configure_sysctl
                install_clamav
                configure_journald
                install_rkhunter
                configure_network
                create_backup_script
                echo -e "${GREEN}Всички операции са изпълнени успешно!${NC}"
                pause
                ;;
            [Qq]) 
                echo -e "${GREEN}Изход от скрипта.${NC}"
                exit 0
                ;;
            *) 
                echo -e "${RED}Невалиден избор. Моля, опитайте отново.${NC}"
                pause
                ;;
        esac
    done
}

# Стартиране на основното меню
main_menu