#!/bin/bash
set -e

# Оптимизиран Docker инсталатор за Ubuntu
# Версия: 3.0 - Надежден и пълен с функции
# Автор: Федя Серафиев
# Сайт: https://urocibg.eu
# Дата: $(date +%Y-%m-%d)

# Конфигурация
CHANNEL="stable"
VERSION=""
DOWNLOAD_URL="https://download.docker.com"
COLOR_GREEN='\033[0;32m'
COLOR_RED='\033[0;31m'
COLOR_RESET='\033[0m'

# Помощна функция за цветен изход
info() {
  echo -e "${COLOR_GREEN}[INFO]${COLOR_RESET} $1"
}

error() {
  echo -e "${COLOR_RED}[ERROR]${COLOR_RESET} $1" >&2
  exit 1
}

# Обработка на параметри
show_help() {
  cat <<EOF
Употреба: $0 [опции]

Опции:
  --version VERSION   Инсталира конкретна версия на Docker
  --channel CHANNEL   Избира канал (stable/test) (по подразбиране: stable)
  --help              Показва този помощен текст

Примери:
  $0                           # Инсталира най-новата стабилна версия
  $0 --version 24.0            # Инсталира Docker версия 24.0
  $0 --channel test            # Инсталира от test канала
EOF
  exit 0
}

while [[ $# -gt 0 ]]; do
  case $1 in
    --version)
      VERSION="$2"
      shift 2
      ;;
    --channel)
      if [[ "$2" != "stable" && "$2" != "test" ]]; then
        error "Невалиден канал. Използвайте 'stable' или 'test'"
      fi
      CHANNEL="$2"
      shift 2
      ;;
    --help)
      show_help
      ;;
    *)
      error "Неизвестен параметър: $1\nИзползвайте --help за помощ"
      ;;
  esac
done

# Проверка за root права
if [[ $EUID -ne 0 ]]; then
  error "Този скрипт трябва да се изпълнява с root права или чрез sudo"
fi

# Проверка за Ubuntu
if ! grep -q "ubuntu" /etc/os-release 2>/dev/null; then
  error "Този скрипт е предназначен само за Ubuntu системи"
fi

# Получаване на версията на Ubuntu
UBUNTU_VERSION=$(grep -oP 'VERSION_CODENAME=\K\w+' /etc/os-release 2>/dev/null || lsb_release -cs 2>/dev/null)

if [[ -z "$UBUNTU_VERSION" ]]; then
  error "Не може да се определи версията на Ubuntu"
fi

info "🐳 Започва инсталация на Docker Engine за Ubuntu $UBUNTU_VERSION ($CHANNEL канал)"

# Проверка за вече инсталиран Docker
if command -v docker &>/dev/null; then
  info "ℹ️  Docker вече е инсталиран. Ще бъде актуализиран..."
  sleep 2
fi

# Премахване на стари версии
info "🧹 Премахване на стари Docker пакети (ако има такива)..."
{
  apt-get remove -y docker docker-engine docker.io containerd runc || true
  apt-get purge -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin || true
} >/dev/null 2>&1

# Актуализиране на системата
info "🔄 Актуализиране на списъка с пакети..."
apt-get update -qq >/dev/null

# Инсталиране на зависимости
info "📦 Инсталиране на необходимите зависимости..."
apt-get install -y -qq \
  ca-certificates \
  curl \
  gnupg \
  lsb-release \
  software-properties-common \
  apt-transport-https >/dev/null

# Добавяне на Docker GPG ключ
info "🔑 Добавяне на официалния Docker GPG ключ..."
mkdir -p /etc/apt/keyrings
curl -fsSL "$DOWNLOAD_URL/linux/ubuntu/gpg" | gpg --dearmor -o /etc/apt/keyrings/docker.gpg
chmod a+r /etc/apt/keyrings/docker.gpg

# Добавяне на Docker хранилище
info "📥 Добавяне на Docker хранилище за Ubuntu $UBUNTU_VERSION..."
echo "deb [arch=$(dpkg --print-architecture) signed-by=/etc/apt/keyrings/docker.gpg] $DOWNLOAD_URL/linux/ubuntu $UBUNTU_VERSION $CHANNEL" | \
  tee /etc/apt/sources.list.d/docker.list >/dev/null

# Актуализиране на пакетния индекс
info "🔄 Актуализиране на списъка с пакети след добавяне на Docker хранилището..."
apt-get update -qq >/dev/null

# Определяне на пакетите за инсталиране
PACKAGES="docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin"

# Ако е зададена конкретна версия
if [[ -n "$VERSION" ]]; then
  info "🔍 Търсене на Docker версия $VERSION..."
  PKG_VERSION=$(apt-cache madison docker-ce | awk -F"|" '{print $2}' | grep "$VERSION" | tr -d ' ' | head -1)
  
  if [[ -z "$PKG_VERSION" ]]; then
    error "Версия $VERSION не е намерена в хранилището!\nНалични версии:\n$(apt-cache madison docker-ce | head -10)"
  fi
  
  PACKAGES="docker-ce=$PKG_VERSION docker-ce-cli=$PKG_VERSION containerd.io docker-buildx-plugin docker-compose-plugin"
  info "🎯 Ще бъде инсталирана версия $PKG_VERSION"
fi

# Инсталиране на Docker
info "🚀 Инсталиране на Docker..."
export DEBIAN_FRONTEND=noninteractive
apt-get install -y -qq $PACKAGES >/dev/null

# Стартиране и активиране на Docker услугата
info "⚙️  Конфигуриране на Docker услугата..."
systemctl enable --now docker >/dev/null 2>&1

# Проверка на инсталацията
DOCKER_VERSION=$(docker --version 2>/dev/null || true)
if [[ -z "$DOCKER_VERSION" ]]; then
  error "Docker не е инсталиран успешно"
else
  info "✅ $DOCKER_VERSION е инсталиран успешно"
fi

# Тестване с hello-world
info "🧪 Тестване на Docker инсталацията..."
if docker run --rm hello-world >/dev/null 2>&1; then
  info "🎉 Docker работи правилно!"
else
  error "Docker инсталацията има проблеми"
fi

# Добавяне на потребителя в docker групата
CURRENT_USER=${SUDO_USER:-$(whoami)}
if ! id -nG "$CURRENT_USER" | grep -qw "docker"; then
  info "👥 Добавяне на потребител '$CURRENT_USER' към docker групата..."
  usermod -aG docker "$CURRENT_USER"
  info "ℹ️  За да влязат промените в сила, изпълнете:\n  ${COLOR_GREEN}newgrp docker${COLOR_RESET}\nили излезте от системата и влезте отново"
fi

# Показване на полезни команди
cat <<EOF

${COLOR_GREEN}🎉 Docker е успешно инсталиран и конфигуриран!${COLOR_RESET}

📝 Полезни команди:
   docker --version                 # Проверка на версията
   docker info                      # Информация за Docker
   docker run hello-world           # Тестване
   docker ps                        # Списък на работещите контейнери
   docker ps -a                     # Списък на всички контейнери
   docker images                    # Списък на изображенията
   docker exec -it <container> bash # Влизане в контейнер

🔧 Управление на услугата:
   sudo systemctl start docker      # Стартиране на Docker
   sudo systemctl stop docker       # Спиране на Docker
   sudo systemctl restart docker    # Рестартиране на Docker

ℹ️  За повече информация посетете: https://docs.docker.com
EOF