#!/bin/bash

# Proxmox Apps Installer
# Професионален инсталатор за Docker приложения в Proxmox контейнери
# Автор: Федя Серафиев
# Версия: 1.0

set -e

# Цветове за по-добра визуализация
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
PURPLE='\033[0;35m'
CYAN='\033[0;36m'
WHITE='\033[1;37m'
NC='\033[0m' # No Color

# Глобални променливи
INSTALL_DIR="/opt/proxmox-apps"
DATA_DIR="/var/lib/proxmox-apps"
LOG_FILE="/var/log/proxmox-apps-installer.log"

# Функция за лог
log() {
    echo "$(date '+%Y-%m-%d %H:%M:%S') - $1" | tee -a "$LOG_FILE"
}

# Функция за показване на header
show_header() {
    clear
    echo -e "${CYAN}╔═══════════════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║                    ${WHITE}PROXMOX APPS INSTALLER${CYAN}                                    ║${NC}"
    echo -e "${CYAN}║                   ${YELLOW}Професионален Docker инсталатор${CYAN}                           ║${NC}"
    echo -e "${CYAN}╠═══════════════════════════════════════════════════════════════════════════════╣${NC}"
    echo -e "${CYAN}║${NC} ${GREEN}✓${NC} Nginx Proxy Manager - Reverse Proxy & SSL Management              ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC} ${GREEN}✓${NC} Dashy V3 - Personalized Dashboard                                  ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC} ${GREEN}✓${NC} File Browser - Web File Manager                                    ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC} ${GREEN}✓${NC} Droppy - Modern File Sharing                                       ${CYAN}║${NC}"
    echo -e "${CYAN}║${NC} ${GREEN}✓${NC} OpenGist - Self-hosted Pastebin                                    ${CYAN}║${NC}"
    echo -e "${CYAN}╚═══════════════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
}

# Функция за почистване на проблемни repositories
fix_repositories() {
    log "Почистване на проблемни repositories..."
    
    # Почистване на AnyDesk repository ако създава проблеми
    if [[ -f /etc/apt/sources.list.d/anydesk-stable.list ]]; then
        echo -e "${YELLOW}⚠️  Намерен AnyDesk repository, премахвам временно...${NC}"
        mv /etc/apt/sources.list.d/anydesk-stable.list /etc/apt/sources.list.d/anydesk-stable.list.bak || true
    fi
    
    # Почистване на кеша
    apt-get clean
    rm -rf /var/lib/apt/lists/*
    
    # Актуализиране на пакетите
    apt-get update -qq
    
    log "Repositories са почистени"
}

# Функция за проверка на системни изисквания
check_requirements() {
    log "Проверка на системни изисквания..."
    
    # Почистване на repositories преди проверка
    fix_repositories
    
    # Проверка дали сме в Proxmox контейнер
    if [[ ! -f /proc/1/cgroup ]] || ! grep -q "lxc" /proc/1/cgroup 2>/dev/null; then
        echo -e "${YELLOW}⚠️  Предупреждение: Не се засича Proxmox LXC контейнер${NC}"
        echo -e "${YELLOW}   Скриптът ще продължи, но е оптимизиран за Proxmox${NC}"
        echo ""
    fi
    
    # Проверка на свободно пространство
    AVAILABLE_SPACE=$(df / | tail -1 | awk '{print $4}')
    if [[ $AVAILABLE_SPACE -lt 5242880 ]]; then # 5GB в KB
        echo -e "${RED}❌ Недостатъчно дисково пространство. Нужни са поне 5GB${NC}"
        exit 1
    fi
    
    # Проверка на мрежова свързаност
    if ! ping -c 1 8.8.8.8 &> /dev/null; then
        echo -e "${RED}❌ Няма интернет връзка${NC}"
        exit 1
    fi
    
    echo -e "${GREEN}✅ Системните изисквания са изпълнени${NC}"
}

# Функция за инсталиране на Docker
install_docker() {
    if command -v docker &> /dev/null; then
        log "Docker вече е инсталиран"
        return 0
    fi
    
    log "Инсталиране на Docker..."
    echo -e "${BLUE}📦 Инсталиране на Docker и Docker Compose...${NC}"
    
    # Актуализиране на пакетите
    apt-get update -qq
    apt-get install -y apt-transport-https ca-certificates curl gnupg lsb-release software-properties-common
    
    # Определяне на правилната Ubuntu версия за Docker
    UBUNTU_CODENAME=$(lsb_release -cs)
    
    # Проверка дали версията е поддържана от Docker
    case $UBUNTU_CODENAME in
        "focal"|"jammy"|"mantic"|"noble")
            DOCKER_CODENAME=$UBUNTU_CODENAME
            ;;
        *)
            # Ако версията не е поддържана, използвай focal като fallback
            echo -e "${YELLOW}⚠️  Ubuntu $UBUNTU_CODENAME не е официално поддържана, използвам focal${NC}"
            DOCKER_CODENAME="focal"
            ;;
    esac
    
    # Премахване на стари Docker инсталации
    apt-get remove -y docker docker-engine docker.io containerd runc || true
    
    # Добавяне на Docker GPG ключ
    curl -fsSL https://download.docker.com/linux/ubuntu/gpg | gpg --dearmor -o /usr/share/keyrings/docker-archive-keyring.gpg
    
    # Добавяне на Docker repository
    echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker-archive-keyring.gpg] https://download.docker.com/linux/ubuntu $DOCKER_CODENAME stable" | tee /etc/apt/sources.list.d/docker.list > /dev/null
    
    # Актуализиране на пакетите
    apt-get update -qq
    
    # Инсталиране на Docker
    apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin
    
    # Проверка на инсталацията
    if ! docker --version &> /dev/null; then
        log "Грешка при инсталирането на Docker"
        exit 1
    fi
    
    # Стартиране на Docker
    systemctl enable docker
    systemctl start docker
    
    # Инсталиране на Docker Compose standalone
    DOCKER_COMPOSE_VERSION=$(curl -s https://api.github.com/repos/docker/compose/releases/latest | grep -oP '"tag_name": "\K(.*)(?=")')
    curl -L "https://github.com/docker/compose/releases/download/${DOCKER_COMPOSE_VERSION}/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
    chmod +x /usr/local/bin/docker-compose
    
    # Проверка на Docker Compose
    if ! docker-compose --version &> /dev/null; then
        log "Грешка при инсталирането на Docker Compose"
        exit 1
    fi
    
    log "Docker е инсталиран успешно"
}

# Функция за създаване на директории
create_directories() {
    log "Създаване на директории..."
    
    mkdir -p "$INSTALL_DIR"
    mkdir -p "$DATA_DIR"
    mkdir -p "$DATA_DIR/nginx-proxy-manager"
    mkdir -p "$DATA_DIR/dashy"
    mkdir -p "$DATA_DIR/filebrowser"
    mkdir -p "$DATA_DIR/droppy"
    mkdir -p "$DATA_DIR/opengist"
    
    log "Директориите са създадени"
}

# Функция за диагностика на проблеми
diagnose_system() {
    echo -e "${BLUE}🔍 Диагностика на системата...${NC}"
    
    echo -e "${YELLOW}📋 Система:${NC}"
    echo "  OS: $(cat /etc/os-release | grep PRETTY_NAME | cut -d= -f2 | tr -d '\"')"
    echo "  Kernel: $(uname -r)"
    echo "  Architecture: $(dpkg --print-architecture)"
    echo ""
    
    echo -e "${YELLOW}📋 Мрежа:${NC}"
    echo "  IP адрес: $(ip route get 8.8.8.8 | sed -n '/src/{s/.*src *\([^ ]*\).*/\1/p;q}' 2>/dev/null || echo 'Неизвестен')"
    echo "  DNS: $(cat /etc/resolv.conf | grep nameserver | head -1 | awk '{print $2}' || echo 'Неизвестен')"
    echo ""
    
    echo -e "${YELLOW}📋 Репозитории:${NC}"
    apt-get update -qq 2>&1 | grep -E "(ERROR|WARNING|W:|E:)" | head -5 || echo "  Няма грешки"
    echo ""
    
    echo -e "${YELLOW}📋 Дисково пространство:${NC}"
    df -h / | tail -1 | awk '{print "  Използвано: " $3 "/" $2 " (" $5 ")"}'
    echo ""
    
    echo -e "${YELLOW}📋 Памет:${NC}"
    free -h | grep Mem | awk '{print "  Използвана: " $3 "/" $2}'
    echo ""
}

# Функция за автоматично поправяне на проблеми
auto_fix() {
    log "Автоматично поправяне на проблеми..."
    
    # Поправяне на времева зона
    if [[ ! -f /etc/timezone ]] || [[ $(cat /etc/timezone) != "Europe/Sofia" ]]; then
        echo -e "${BLUE}🕐 Настройка на времева зона...${NC}"
        ln -sf /usr/share/zoneinfo/Europe/Sofia /etc/localtime
        echo "Europe/Sofia" > /etc/timezone
        dpkg-reconfigure -f noninteractive tzdata
    fi
    
    # Поправяне на locale
    if ! locale -a | grep -q "bg_BG.UTF-8"; then
        echo -e "${BLUE}🌍 Инсталиране на български locale...${NC}"
        apt-get install -y locales
        locale-gen bg_BG.UTF-8
        update-locale LANG=bg_BG.UTF-8
    fi
    
    # Поправяне на DNS
    if ! grep -q "8.8.8.8" /etc/resolv.conf; then
        echo -e "${BLUE}🌐 Добавяне на резервен DNS...${NC}"
        echo "nameserver 8.8.8.8" >> /etc/resolv.conf
        echo "nameserver 8.8.4.4" >> /etc/resolv.conf
    fi
    
    log "Автоматичното поправяне е завършено"
}

# Функция за генериране на случайни пароли
generate_password() {
    openssl rand -base64 32 | tr -d "=+/" | cut -c1-25
}

# Функция за инсталиране на Nginx Proxy Manager
install_nginx_proxy_manager() {
    log "Инсталиране на Nginx Proxy Manager..."
    
    cat > "$INSTALL_DIR/nginx-proxy-manager.yml" << 'EOF'
version: '3.8'
services:
  nginx-proxy-manager:
    image: 'jc21/nginx-proxy-manager:latest'
    container_name: nginx-proxy-manager
    restart: unless-stopped
    ports:
      - '80:80'
      - '443:443'
      - '81:81'
    volumes:
      - /var/lib/proxmox-apps/nginx-proxy-manager:/data
      - /var/lib/proxmox-apps/nginx-proxy-manager/letsencrypt:/etc/letsencrypt
    environment:
      - DISABLE_IPV6=true
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:81/api/"]
      interval: 30s
      timeout: 10s
      retries: 3

networks:
  proxy-network:
    name: proxy-network
    driver: bridge
EOF
    
    cd "$INSTALL_DIR"
    docker-compose -f nginx-proxy-manager.yml up -d
    
    echo -e "${GREEN}✅ Nginx Proxy Manager е инсталиран${NC}"
    echo -e "${YELLOW}📋 Администриране: http://your-ip:81${NC}"
    echo -e "${YELLOW}📋 Потребител: admin@example.com${NC}"
    echo -e "${YELLOW}📋 Парола: changeme${NC}"
    echo ""
}

# Функция за инсталиране на Dashy V3
install_dashy() {
    log "Инсталиране на Dashy V3..."
    
    # Създаване на базова конфигурация
    cat > "$DATA_DIR/dashy/conf.yml" << 'EOF'
pageInfo:
  title: 'Proxmox Dashboard'
  description: 'Персонализиран Dashboard за Proxmox Apps'
  logo: 'https://raw.githubusercontent.com/Lissy93/dashy/master/public/img/dashy.png'
  
appConfig:
  theme: 'dark'
  layout: 'auto'
  iconSize: 'medium'
  language: 'bg'
  
sections:
  - name: 'Proxmox Services'
    icon: 'fas fa-server'
    items:
      - title: 'Nginx Proxy Manager'
        description: 'Reverse Proxy & SSL Management'
        url: 'http://localhost:81'
        icon: 'fas fa-shield-alt'
      - title: 'File Browser'
        description: 'Web File Manager'
        url: 'http://localhost:8080'
        icon: 'fas fa-folder'
      - title: 'Droppy'
        description: 'File Sharing'
        url: 'http://localhost:8989'
        icon: 'fas fa-cloud-upload-alt'
      - title: 'OpenGist'
        description: 'Code Snippets'
        url: 'http://localhost:6157'
        icon: 'fas fa-code'
        
  - name: 'System Info'
    icon: 'fas fa-info-circle'
    widgets:
      - type: 'system-info'
        options:
          hostname: true
          uptime: true
          memory: true
          cpu: true
EOF
    
    cat > "$INSTALL_DIR/dashy.yml" << 'EOF'
version: '3.8'
services:
  dashy:
    image: 'lissy93/dashy:latest'
    container_name: dashy
    restart: unless-stopped
    ports:
      - '4000:80'
    volumes:
      - /var/lib/proxmox-apps/dashy/conf.yml:/app/public/conf.yml:ro
    environment:
      - NODE_ENV=production
      - UID=1000
      - GID=1000
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:80"]
      interval: 30s
      timeout: 10s
      retries: 3

networks:
  proxy-network:
    external: true
EOF
    
    cd "$INSTALL_DIR"
    docker-compose -f dashy.yml up -d
    
    echo -e "${GREEN}✅ Dashy V3 е инсталиран${NC}"
    echo -e "${YELLOW}📋 Достъп: http://your-ip:4000${NC}"
    echo ""
}

# Функция за инсталиране на File Browser
install_filebrowser() {
    log "Инсталиране на File Browser..."
    
    # Създаване на база данни
    touch "$DATA_DIR/filebrowser/database.db"
    
    cat > "$INSTALL_DIR/filebrowser.yml" << 'EOF'
version: '3.8'
services:
  filebrowser:
    image: 'filebrowser/filebrowser:latest'
    container_name: filebrowser
    restart: unless-stopped
    ports:
      - '8080:80'
    volumes:
      - /var/lib/proxmox-apps/filebrowser/database.db:/database.db
      - /var/lib/proxmox-apps:/srv/proxmox-apps
      - /opt/proxmox-apps:/srv/configs
    environment:
      - FB_DATABASE=/database.db
      - FB_ROOT=/srv
      - FB_LOG=stdout
      - FB_NOAUTH=false
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:80/health"]
      interval: 30s
      timeout: 10s
      retries: 3

networks:
  proxy-network:
    external: true
EOF
    
    cd "$INSTALL_DIR"
    docker-compose -f filebrowser.yml up -d
    
    echo -e "${GREEN}✅ File Browser е инсталиран${NC}"
    echo -e "${YELLOW}📋 Достъп: http://your-ip:8080${NC}"
    echo -e "${YELLOW}📋 Потребител: admin${NC}"
    echo -e "${YELLOW}📋 Парола: admin${NC}"
    echo ""
}

# Функция за инсталиране на Droppy
install_droppy() {
    log "Инсталиране на Droppy..."
    
    mkdir -p "$DATA_DIR/droppy/config"
    mkdir -p "$DATA_DIR/droppy/files"
    
    cat > "$INSTALL_DIR/droppy.yml" << 'EOF'
version: '3.8'
services:
  droppy:
    image: 'silverwind/droppy:latest'
    container_name: droppy
    restart: unless-stopped
    ports:
      - '8989:8989'
    volumes:
      - /var/lib/proxmox-apps/droppy/config:/app/config
      - /var/lib/proxmox-apps/droppy/files:/app/files
    environment:
      - NODE_ENV=production
      - UID=1000
      - GID=1000
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8989"]
      interval: 30s
      timeout: 10s
      retries: 3

networks:
  proxy-network:
    external: true
EOF
    
    cd "$INSTALL_DIR"
    docker-compose -f droppy.yml up -d
    
    echo -e "${GREEN}✅ Droppy е инсталиран${NC}"
    echo -e "${YELLOW}📋 Достъп: http://your-ip:8989${NC}"
    echo ""
}

# Функция за инсталиране на OpenGist
install_opengist() {
    log "Инсталиране на OpenGist..."
    
    OPENGIST_SECRET=$(generate_password)
    
    cat > "$INSTALL_DIR/opengist.yml" << EOF
version: '3.8'
services:
  opengist:
    image: 'ghcr.io/thomiceli/opengist:latest'
    container_name: opengist
    restart: unless-stopped
    ports:
      - '6157:6157'
    volumes:
      - /var/lib/proxmox-apps/opengist:/opengist
    environment:
      - OG_DB_TYPE=sqlite
      - OG_DB_PATH=/opengist/opengist.db
      - OG_SECRET_KEY=${OPENGIST_SECRET}
      - OG_EXTERNAL_URL=http://localhost:6157
      - OG_LOG_LEVEL=info
      - OG_DISABLE_SIGNUP=false
      - OG_REQUIRE_LOGIN=false
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:6157"]
      interval: 30s
      timeout: 10s
      retries: 3

networks:
  proxy-network:
    external: true
EOF
    
    cd "$INSTALL_DIR"
    docker-compose -f opengist.yml up -d
    
    echo -e "${GREEN}✅ OpenGist е инсталиран${NC}"
    echo -e "${YELLOW}📋 Достъп: http://your-ip:6157${NC}"
    echo ""
}

# Функция за създаване на master Docker Compose файл
create_master_compose() {
    log "Създаване на master Docker Compose файл..."
    
    cat > "$INSTALL_DIR/docker-compose.yml" << 'EOF'
version: '3.8'

services:
  # Nginx Proxy Manager
  nginx-proxy-manager:
    image: 'jc21/nginx-proxy-manager:latest'
    container_name: nginx-proxy-manager
    restart: unless-stopped
    ports:
      - '80:80'
      - '443:443'
      - '81:81'
    volumes:
      - /var/lib/proxmox-apps/nginx-proxy-manager:/data
      - /var/lib/proxmox-apps/nginx-proxy-manager/letsencrypt:/etc/letsencrypt
    environment:
      - DISABLE_IPV6=true
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:81/api/"]
      interval: 30s
      timeout: 10s
      retries: 3

  # Dashy Dashboard
  dashy:
    image: 'lissy93/dashy:latest'
    container_name: dashy
    restart: unless-stopped
    ports:
      - '4000:80'
    volumes:
      - /var/lib/proxmox-apps/dashy/conf.yml:/app/public/conf.yml:ro
    environment:
      - NODE_ENV=production
      - UID=1000
      - GID=1000
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:80"]
      interval: 30s
      timeout: 10s
      retries: 3

  # File Browser
  filebrowser:
    image: 'filebrowser/filebrowser:latest'
    container_name: filebrowser
    restart: unless-stopped
    ports:
      - '8080:80'
    volumes:
      - /var/lib/proxmox-apps/filebrowser/database.db:/database.db
      - /var/lib/proxmox-apps:/srv/proxmox-apps
      - /opt/proxmox-apps:/srv/configs
    environment:
      - FB_DATABASE=/database.db
      - FB_ROOT=/srv
      - FB_LOG=stdout
      - FB_NOAUTH=false
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:80/health"]
      interval: 30s
      timeout: 10s
      retries: 3

  # Droppy File Sharing
  droppy:
    image: 'silverwind/droppy:latest'
    container_name: droppy
    restart: unless-stopped
    ports:
      - '8989:8989'
    volumes:
      - /var/lib/proxmox-apps/droppy/config:/app/config
      - /var/lib/proxmox-apps/droppy/files:/app/files
    environment:
      - NODE_ENV=production
      - UID=1000
      - GID=1000
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:8989"]
      interval: 30s
      timeout: 10s
      retries: 3

  # OpenGist
  opengist:
    image: 'ghcr.io/thomiceli/opengist:latest'
    container_name: opengist
    restart: unless-stopped
    ports:
      - '6157:6157'
    volumes:
      - /var/lib/proxmox-apps/opengist:/opengist
    environment:
      - OG_DB_TYPE=sqlite
      - OG_DB_PATH=/opengist/opengist.db
      - OG_SECRET_KEY=CHANGE_THIS_SECRET_KEY
      - OG_EXTERNAL_URL=http://localhost:6157
      - OG_LOG_LEVEL=info
      - OG_DISABLE_SIGNUP=false
      - OG_REQUIRE_LOGIN=false
    networks:
      - proxy-network
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:6157"]
      interval: 30s
      timeout: 10s
      retries: 3

networks:
  proxy-network:
    name: proxy-network
    driver: bridge
EOF
    
    # Заместване на placeholder-a за OpenGist secret
    OPENGIST_SECRET=$(generate_password)
    sed -i "s/CHANGE_THIS_SECRET_KEY/${OPENGIST_SECRET}/g" "$INSTALL_DIR/docker-compose.yml"
    
    log "Master Docker Compose файлът е създаден"
}

# Функция за създаване на управляващи скриптове
create_management_scripts() {
    log "Създаване на управляващи скриптове..."
    
    # Скрипт за старт на всички услуги
    cat > "$INSTALL_DIR/start-all.sh" << 'EOF'
#!/bin/bash
cd /opt/proxmox-apps
echo "Стартиране на всички услуги..."
docker-compose up -d
echo "Всички услуги са стартирани!"
EOF
    
    # Скрипт за спиране на всички услуги
    cat > "$INSTALL_DIR/stop-all.sh" << 'EOF'
#!/bin/bash
cd /opt/proxmox-apps
echo "Спиране на всички услуги..."
docker-compose down
echo "Всички услуги са спрени!"
EOF
    
    # Скрипт за рестарт на всички услуги
    cat > "$INSTALL_DIR/restart-all.sh" << 'EOF'
#!/bin/bash
cd /opt/proxmox-apps
echo "Рестартиране на всички услуги..."
docker-compose down
docker-compose up -d
echo "Всички услуги са рестартирани!"
EOF
    
    # Скрипт за проверка на статуса
    cat > "$INSTALL_DIR/status.sh" << 'EOF'
#!/bin/bash
cd /opt/proxmox-apps
echo "Статус на услугите:"
docker-compose ps
echo ""
echo "Използване на ресурси:"
docker stats --no-stream
EOF
    
    # Скрипт за бекъп
    cat > "$INSTALL_DIR/backup.sh" << 'EOF'
#!/bin/bash
BACKUP_DIR="/var/backups/proxmox-apps"
DATE=$(date +%Y%m%d_%H%M%S)
BACKUP_FILE="$BACKUP_DIR/proxmox-apps-backup-$DATE.tar.gz"

mkdir -p "$BACKUP_DIR"

echo "Създаване на бекъп..."
tar -czf "$BACKUP_FILE" -C /var/lib proxmox-apps -C /opt proxmox-apps

echo "Бекъпът е създаден: $BACKUP_FILE"
ls -lh "$BACKUP_FILE"
EOF
    
    # Скрипт за актуализиране
    cat > "$INSTALL_DIR/update.sh" << 'EOF'
#!/bin/bash
cd /opt/proxmox-apps
echo "Актуализиране на Docker образите..."
docker-compose pull
docker-compose up -d
echo "Премахване на стари образи..."
docker image prune -f
echo "Актуализирането е завършено!"
EOF
    
    # Правене на скриптовете изпълними
    chmod +x "$INSTALL_DIR"/*.sh
    
    log "Управляващите скриптове са създадени"
}

# Функция за показване на меню
show_menu() {
    show_header
    echo -e "${WHITE}Изберете опция:${NC}"
    echo ""
    echo -e "${GREEN}1)${NC} Инсталиране на Nginx Proxy Manager"
    echo -e "${GREEN}2)${NC} Инсталиране на Dashy V3"
    echo -e "${GREEN}3)${NC} Инсталиране на File Browser"
    echo -e "${GREEN}4)${NC} Инсталиране на Droppy"
    echo -e "${GREEN}5)${NC} Инсталиране на OpenGist"
    echo -e "${GREEN}6)${NC} Инсталиране на всички приложения"
    echo -e "${GREEN}7)${NC} Показване на статус"
    echo -e "${GREEN}8)${NC} Рестарт на всички услуги"
    echo -e "${GREEN}9)${NC} Спиране на всички услуги"
    echo -e "${GREEN}10)${NC} Създаване на бекъп"
    echo -e "${GREEN}11)${NC} Актуализиране на приложенията"
    echo -e "${RED}0)${NC} Изход"
    echo ""
    echo -e "${YELLOW}Моля, въведете вашия избор [0-11]:${NC} "
}

# Главна функция
main() {
    # Създаване на лог файл
    touch "$LOG_FILE"
    log "Стартиране на Proxmox Apps Installer"
    
    # Проверка дали скриптът се изпълнява като root
    if [[ $EUID -ne 0 ]]; then
        echo -e "${RED}❌ Този скрипт трябва да се изпълни като root${NC}"
        exit 1
    fi
    
    # Проверка на системни изисквания
    check_requirements
    
    # Инсталиране на Docker ако не е инсталиран
    install_docker
    
    # Създаване на директории
    create_directories
    
    # Главно меню
    while true; do
        show_menu
        read -r choice
        
        case $choice in
            1)
                echo -e "${BLUE}Инсталиране на Nginx Proxy Manager...${NC}"
                install_nginx_proxy_manager
                read -p "Натиснете Enter за да продължите..."
                ;;
            2)
                echo -e "${BLUE}Инсталиране на Dashy V3...${NC}"
                install_dashy
                read -p "Натиснете Enter за да продължите..."
                ;;
            3)
                echo -e "${BLUE}Инсталиране на File Browser...${NC}"
                install_filebrowser
                read -p "Натиснете Enter за да продължите..."
                ;;
            4)
                echo -e "${BLUE}Инсталиране на Droppy...${NC}"
                install_droppy
                read -p "Натиснете Enter за да продължите..."
                ;;
            5)
                echo -e "${BLUE}Инсталиране на OpenGist...${NC}"
                install_opengist
                read -p "Натиснете Enter за да продължите..."
                ;;
            6)
                echo -e "${BLUE}Инсталиране на всички приложения...${NC}"
                create_master_compose
                cd "$INSTALL_DIR"
                docker-compose up -d
                create_management_scripts
                echo -e "${GREEN}✅ Всички приложения са инсталирани успешно!${NC}"
                echo ""
                echo -e "${YELLOW}📋 Достъп до приложенията:${NC}"
                echo -e "${CYAN}  • Nginx Proxy Manager: http://your-ip:81${NC}"
                echo -e "${CYAN}  • Dashy Dashboard: http://your-ip:4000${NC}"
                echo -e "${CYAN}  • File Browser: http://your-ip:8080${NC}"
                echo -e "${CYAN}  • Droppy: http://your-ip:8989${NC}"
                echo -e "${CYAN}  • OpenGist: http://your-ip:6157${NC}"
                echo ""
                echo -e "${YELLOW}🔧 Управляващи скриптове в $INSTALL_DIR:${NC}"
                echo -e "${CYAN}  • ./start-all.sh - Стартиране на всички услуги${NC}"
                echo -e "${CYAN}  • ./stop-all.sh - Спиране на всички услуги${NC}"
                echo -e "${CYAN}  • ./restart-all.sh - Рестарт на всички услуги${NC}"
                echo -e "${CYAN}  • ./status.sh - Проверка на статуса${NC}"
                echo -e "${CYAN}  • ./backup.sh - Създаване на бекъп${NC}"
                echo -e "${CYAN}  • ./update.sh - Актуализиране${NC}"
                read -p "Натиснете Enter за да продължите..."
                ;;
            7)
                echo -e "${BLUE}Показване на статус...${NC}"
                if [[ -f "$INSTALL_DIR/status.sh" ]]; then
                    bash "$INSTALL_DIR/status.sh"
                else
                    echo -e "${RED}❌ Статус скриптът не е намерен${NC}"
                fi
                read -p "Натиснете Enter за да продължите..."
                ;;
            8)
                echo -e "${BLUE}Рестарт на всички услуги...${NC}"
                if [[ -f "$INSTALL_DIR/restart-all.sh" ]]; then
                    bash "$INSTALL_DIR/restart-all.sh"
                else
                    echo -e "${RED}❌ Рестарт скриптът не е намерен${NC}"
                fi
                read -p "Натиснете Enter за да продължите..."
                ;;
            9)
                echo -e "${BLUE}Спиране на всички услуги...${NC}"
                if [[ -f "$INSTALL_DIR/stop-all.sh" ]]; then
                    bash "$INSTALL_DIR/stop-all.sh"
                else
                    echo -e "${RED}❌ Стоп скриптът не е намерен${NC}"
                fi
                read -p "Натиснете Enter за да продължите..."
                ;;
            10)
                echo -e "${BLUE}Създаване на бекъп...${NC}"
                if [[ -f "$INSTALL_DIR/backup.sh" ]]; then
                    bash "$INSTALL_DIR/backup.sh"
                else
                    echo -e "${RED}❌ Бекъп скриптът не е намерен${NC}"
                fi
                read -p "Натиснете Enter за да продължите..."
                ;;
            11)
                echo -e "${BLUE}Актуализиране на приложенията...${NC}"
                if [[ -f "$INSTALL_DIR/update.sh" ]]; then
                    bash "$INSTALL_DIR/update.sh"
                else
                    echo -e "${RED}❌ Актуализиращият скрипт не е намерен${NC}"
                fi
                read -p "Натиснете Enter за да продължите..."
                ;;
            12)
                echo -e "${BLUE}Диагностика на системата...${NC}"
                diagnose_system
                read -p "Натиснете Enter за да продължите..."
                ;;
            13)
                echo -e "${BLUE}Автоматично поправяне...${NC}"
                auto_fix
                echo -e "${GREEN}✅ Автоматичното поправяне е завършено${NC}"
                read -p "Натиснете Enter за да продължите..."
                ;;
            0)
                echo -e "${GREEN}Благодаря за използването на Proxmox Apps Installer!${NC}"
                log "Инсталаторът е приключен"
                exit 0
                ;;
            *)
                echo -e "${RED}❌ Невалиден избор. Моля, опитайте отново.${NC}"
                read -p "Натиснете Enter за да продължите..."
                ;;
        esac
    done
}

# Функция за показване на информация за системата
show_system_info() {
    echo -e "${CYAN}╔═══════════════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║                            ${WHITE}СИСТЕМНА ИНФОРМАЦИЯ${CYAN}                                ║${NC}"
    echo -e "${CYAN}╚═══════════════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "${YELLOW}📊 Система:${NC} $(hostnamectl | grep "Operating System" | cut -d: -f2 | xargs)"
    echo -e "${YELLOW}🖥️  Хост:${NC} $(hostname)"
    echo -e "${YELLOW}💾 Памет:${NC} $(free -h | grep Mem | awk '{print $3"/"$2}')"
    echo -e "${YELLOW}💿 Диск:${NC} $(df -h / | tail -1 | awk '{print $3"/"$2" ("$5" използвани)"}')"
    echo -e "${YELLOW}🐳 Docker:${NC} $(docker --version 2>/dev/null || echo "Не е инсталиран")"
    echo -e "${YELLOW}🔧 Docker Compose:${NC} $(docker-compose --version 2>/dev/null || echo "Не е инсталиран")"
    echo ""
}

# Функция за проверка на портове
check_ports() {
    echo -e "${BLUE}Проверка на портове...${NC}"
    
    declare -A PORTS=(
        ["80"]="HTTP (Nginx Proxy Manager)"
        ["443"]="HTTPS (Nginx Proxy Manager)"
        ["81"]="Nginx Proxy Manager Admin"
        ["4000"]="Dashy Dashboard"
        ["8080"]="File Browser"
        ["8989"]="Droppy"
        ["6157"]="OpenGist"
    )
    
    for port in "${!PORTS[@]}"; do
        if ss -tuln | grep -q ":$port "; then
            echo -e "${RED}❌ Порт $port (${PORTS[$port]}) е зает${NC}"
        else
            echo -e "${GREEN}✅ Порт $port (${PORTS[$port]}) е свободен${NC}"
        fi
    done
    echo ""
}

# Функция за firewall конфигурация
configure_firewall() {
    log "Конфигуриране на firewall..."
    
    if command -v ufw &> /dev/null; then
        echo -e "${BLUE}Конфигуриране на UFW firewall...${NC}"
        
        # Разрешаване на портове
        ufw allow 80/tcp comment "HTTP"
        ufw allow 443/tcp comment "HTTPS"
        ufw allow 81/tcp comment "Nginx Proxy Manager"
        ufw allow 4000/tcp comment "Dashy"
        ufw allow 8080/tcp comment "File Browser"
        ufw allow 8989/tcp comment "Droppy"
        ufw allow 6157/tcp comment "OpenGist"
        
        # Разрешаване на SSH
        ufw allow 22/tcp comment "SSH"
        
        echo -e "${GREEN}✅ UFW firewall е конфигуриран${NC}"
    else
        echo -e "${YELLOW}⚠️  UFW не е инсталиран. Моля, конфигурирайте firewall-а ръчно${NC}"
    fi
}

# Функция за показване на полезни команди
show_useful_commands() {
    echo -e "${CYAN}╔═══════════════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║                           ${WHITE}ПОЛЕЗНИ КОМАНДИ${CYAN}                                    ║${NC}"
    echo -e "${CYAN}╚═══════════════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "${YELLOW}🐳 Docker команди:${NC}"
    echo -e "${CYAN}  docker ps${NC} - Показва работещите контейнери"
    echo -e "${CYAN}  docker logs <container-name>${NC} - Показва логове на контейнер"
    echo -e "${CYAN}  docker exec -it <container-name> /bin/bash${NC} - Влизане в контейнер"
    echo -e "${CYAN}  docker system prune -f${NC} - Почистване на неизползвани ресурси"
    echo ""
    echo -e "${YELLOW}🔧 Docker Compose команди:${NC}"
    echo -e "${CYAN}  docker-compose ps${NC} - Статус на услугите"
    echo -e "${CYAN}  docker-compose logs -f${NC} - Следене на логове"
    echo -e "${CYAN}  docker-compose pull${NC} - Актуализиране на образите"
    echo -e "${CYAN}  docker-compose restart <service>${NC} - Рестарт на услуга"
    echo ""
    echo -e "${YELLOW}📁 Важни директории:${NC}"
    echo -e "${CYAN}  $INSTALL_DIR${NC} - Конфигурационни файлове"
    echo -e "${CYAN}  $DATA_DIR${NC} - Данни на приложенията"
    echo -e "${CYAN}  $LOG_FILE${NC} - Лог файл на инсталатора"
    echo ""
}

# Функция за генериране на SSL сертификати
generate_ssl_info() {
    echo -e "${CYAN}╔═══════════════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║                          ${WHITE}SSL СЕРТИФИКАТИ${CYAN}                                     ║${NC}"
    echo -e "${CYAN}╚═══════════════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "${YELLOW}🔐 За SSL сертификати използвайте Nginx Proxy Manager:${NC}"
    echo -e "${CYAN}  1. Отворете http://your-ip:81${NC}"
    echo -e "${CYAN}  2. Влезте с admin@example.com / changeme${NC}"
    echo -e "${CYAN}  3. Променете паролата при първо влизане${NC}"
    echo -e "${CYAN}  4. Добавете домейни в 'Proxy Hosts'${NC}"
    echo -e "${CYAN}  5. Активирайте 'Force SSL' и 'HTTP/2 Support'${NC}"
    echo -e "${CYAN}  6. Nginx Proxy Manager автоматично ще генерира Let's Encrypt сертификати${NC}"
    echo ""
}

# Функция за създаване на systemd службa
create_systemd_service() {
    log "Създаване на systemd служба..."
    
    cat > "/etc/systemd/system/proxmox-apps.service" << 'EOF'
[Unit]
Description=Proxmox Apps Docker Compose
Requires=docker.service
After=docker.service

[Service]
Type=forking
RemainAfterExit=yes
WorkingDirectory=/opt/proxmox-apps
ExecStart=/usr/local/bin/docker-compose up -d
ExecStop=/usr/local/bin/docker-compose down
ExecReload=/usr/local/bin/docker-compose restart
TimeoutStartSec=0

[Install]
WantedBy=multi-user.target
EOF
    
    systemctl daemon-reload
    systemctl enable proxmox-apps.service
    
    echo -e "${GREEN}✅ Systemd служба е създадена и активирана${NC}"
    echo -e "${YELLOW}📋 Използвайте: systemctl start|stop|restart proxmox-apps${NC}"
}

# Trap функция за почистване при прекъсване
cleanup() {
    log "Почистване при прекъсване..."
    echo -e "${YELLOW}⚠️  Инсталацията е прекъсната${NC}"
    exit 130
}

# Настройка на trap за SIGINT (Ctrl+C)
trap cleanup SIGINT

# Функция за показване на help
show_help() {
    echo -e "${CYAN}╔═══════════════════════════════════════════════════════════════════════════════╗${NC}"
    echo -e "${CYAN}║                        ${WHITE}ПОМОЩНА ИНФОРМАЦИЯ${CYAN}                                   ║${NC}"
    echo -e "${CYAN}╚═══════════════════════════════════════════════════════════════════════════════╝${NC}"
    echo ""
    echo -e "${YELLOW}📖 Използване:${NC}"
    echo -e "${CYAN}  sudo ./proxmox-apps-installer.sh${NC}"
    echo ""
    echo -e "${YELLOW}🔧 Опции:${NC}"
    echo -e "${CYAN}  --help${NC} - Показва тази помощна информация"
    echo -e "${CYAN}  --version${NC} - Показва версията на скрипта"
    echo -e "${CYAN}  --check${NC} - Проверява системните изисквания"
    echo -e "${CYAN}  --ports${NC} - Проверява наличността на портове"
    echo ""
    echo -e "${YELLOW}📂 Структура на файлове:${NC}"
    echo -e "${CYAN}  /opt/proxmox-apps/${NC} - Конфигурационни файлове"
    echo -e "${CYAN}  /var/lib/proxmox-apps/${NC} - Данни на приложенията"
    echo -e "${CYAN}  /var/log/proxmox-apps-installer.log${NC} - Лог файл"
    echo ""
    echo -e "${YELLOW}🌐 Портове по подразбиране:${NC}"
    echo -e "${CYAN}  80/443 - Nginx Proxy Manager (HTTP/HTTPS)${NC}"
    echo -e "${CYAN}  81 - Nginx Proxy Manager Admin${NC}"
    echo -e "${CYAN}  4000 - Dashy Dashboard${NC}"
    echo -e "${CYAN}  8080 - File Browser${NC}"
    echo -e "${CYAN}  8989 - Droppy${NC}"
    echo -e "${CYAN}  6157 - OpenGist${NC}"
    echo ""
}

# Проверка на command line аргументи
case "${1:-}" in
    --help|-h)
        show_help
        exit 0
        ;;
    --version|-v)
        echo "Proxmox Apps Installer v1.0"
        exit 0
        ;;
    --check|-c)
        show_system_info
        check_requirements
        exit 0
        ;;
    --ports|-p)
        check_ports
        exit 0
        ;;
    --diagnose|-d)
        diagnose_system
        exit 0
        ;;
    --fix|-f)
        auto_fix
        exit 0
        ;;
    *)
        # Нормално изпълнение
        main "$@"
        ;;
esac